/* Source for:
 * Focaltech 5x06 touch screen controller.
 * TrueTouch(TM) Standard Product I2C touchscreen driver. This driver can be found here:
 * drivers/input/touchscreen/cyttsp-i2c.c
 *
 * drivers/input/touchscreen/ft5x06-i2c.c
 *
 * Copyright (C) 2009, 2010 Cypress Semiconductor, Inc.
 *
 * Copyright (c) 2010 Barnes & Noble.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2, and only version 2, as published by the
 * Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 */
#include <linux/delay.h>
#include <linux/init.h>
#include <linux/module.h>
#include <linux/i2c.h>
#include <linux/input.h>
#include <linux/slab.h>
#include <linux/gpio.h>
#include <linux/irq.h>
#include <linux/interrupt.h>
#include <linux/timer.h>
#include <linux/workqueue.h>
#include <linux/earlysuspend.h>
#include <linux/firmware.h>
#include <linux/regulator/consumer.h>
//#include <linux/input/ft5x06.h>
//#include "ft5x06_i2c_defs.h"
#include <linux/ft5x06_i2c_defs.h>
#include "ft5x06_ts_i2c.h"

#include <linux/crestron.h>

#define MAX_TOUCH_POINTS_SUPPORTED 2
#define PEN_DOWN	1
#define PEN_UP		0


#define FT5x06_CPTM_ID_COMPANY        0x79
#define FT5x06_CPTM_ID_PRODUCT        0x03

#define FT5x06_NUM_TX                   28
#define FT5x06_NUM_RX                   16

#define FT5x06_PACKET_LENGTH           128
#define FT5x06_VERIFY_NUM_TRIES         10
#define FT5x06_CALIBRATION_NUM_BYTES     8

#define FT5x06_SCAN_DELAY               20

#define FT5x06_CROSSTALK_TEST_TYPE_EVEN  0
#define FT5x06_CROSSTALK_TEST_TYPE_ODD   1

#define FT5x06_ERR_NOT_FACTORY_MODE      1
#define FT5x06_ERR_NOT_WORKING_MODE      2
#define FT5x06_ERR_SCAN_NOT_DONE         3
#define FT5x06_ERR_INVALID_ID            4
#define FT5x06_ERR_INVALID_CHECKSUM      5

#define FT5x06_DRIVER_VERSION       0x0001

#define FT5x06_UPGRADEVER_REG		0xCD

#define FT5x06_DEBUG_VERBOSE             1

#define FT5x06_DEBUG(X)
//#define FT5x06_DEBUG(X) X

// Values of the "event" field in the data from a touch-panel controller.
typedef enum
{
	EP0xx0M06_TOUCH_DOWN,
	EP0xx0M06_TOUCH_UP,
	EP0xx0M06_TOUCH_ON,
	EP0xx0M06_RESERVED
} EP0xx0M06_EVENT_STATES;

// Return the 2 bits representing event.
#define GET_NUM_TOUCHES(status) (status & 0x0f)
#define GET_EVENT(p) ((p.x_h & 0xc0) >> 6)
// Pull the 2 bits representing the touchId.
#define GET_TOUCH_ID(p) ((p.y_h & 0xf0) >> 4)
// Combine the two values into the proper coordinate.
#define GET_COORDINATE(l,h) ((l | (( h & 0x0F)<<8)))
// Gets the coordinates from the ft506_tch structure.
#define GET_XCOORD(p) (GET_COORDINATE(p.x_l, p.x_h))
#define GET_YCOORD(p) (GET_COORDINATE(p.y_l, p.y_h))

/**
 * 1050 defines.
 */
#define MAX_NUM_PACKETS_READ_1050 5
#define DELAY_PACKET_READ_1050          60      // Delay in microseconds between consecutive read of I2C

#define GET_PACKET_ID_1050(buf)  ((buf)[0])
#define GET_PACKET_VALID_1050(buf) ((buf)[1] >> 7)
#define GET_EVENT_1050(buf) ((buf)[1] & 0x01)
#define GET_TOUCH_ID_1050(buf) ((int) (((buf)[1] & 0x7C) >> 2))

/**
 * Kind of hairy, but this is used to bring the X coordinate into the proper
 * range when GET_XCOORD_1050 is used.
 */
#define CALIBRATE_VAL_X 108
#define TSW1050_CALIBRATE_X(x) ((u16)(((x)*CALIBRATE_VAL_X)/100)) //(need to accomodate for Vbutton area)

// Divide coordinates by 16 (32765 original), manipulate x to accomodate for VButton area.
#define GET_XCOORD_1050(buf) ( TSW1050_CALIBRATE_X(((buf)[3] << 8 | (buf)[2]) >> 4) )
#define GET_YCOORD_1050(buf) ((buf[5] << 8 | buf[4]) >> 4)
#define GET_ZCOORD_1050(buf) ((buf[7] << 8 | buf[6]) >> 4)

// Minimum delay between packet reads for the 1050, in us.
#define PACKET_READ_DELAY_1050 (60)


int gTouchTmajor=0xe;     //touch is with finger
int gTouchTminor=0;
int gTouchWmajor=1;      //width is with tool
int gTouchWminor=0;

// end if tsw1050

#define CAP_TS_RST_N                    (2*32 + 3)      /* GPIO_3_3 */
#define CAP_TS_INT_N                    (2*32 + 4)      /* GPIO_3_4 */

u8 tmpint=1;

extern int getProductId(void);
extern int getProductRev(void);
extern int uiTouchIntfInit(struct ft5x06* ts);
extern int uiVbuttonIntfInit(void);

extern unsigned int gVbKeyStates;
struct mutex *keyStateMutex; //mutex lock for gVbKeyStates

extern int g_virt_dbg_keys;

#pragma pack(push,1) // sets the alignment to 1 byte, push saves the alignment
/* Defines the hardware structure of a point. */
typedef struct {
	unsigned char x_h;
	unsigned char x_l;
	unsigned char y_h;
	unsigned char y_l;
	unsigned char res1;
	unsigned char res2;
} ft506_tch;

/* Structure of the touch data read from the hardware. */
typedef struct ft506 {
    unsigned char mode;
    unsigned char gesture_id;
    unsigned char status;
    ft506_tch point[MAX_TOUCH_POINTS_SUPPORTED];
} ft506_data;
#pragma pack(pop) // return to previous byte alignment

/* maintain touch ID */
extern unsigned long gTouchID[MAX_TOUCH_POINTS_SUPPORTED];

extern struct UiHalTouch;

extern struct UiHalTouch uiTouchData[];
extern int uiMultiTouch;

/*****************************************************************************
 * Function Prototypes
 ****************************************************************************/

static void ft5x06_xy_worker(struct work_struct *work);

static irqreturn_t ft5x06_irq(int irq, void *handle);

static int  ft5x06_inlist(u16 prev_track[], u8 cur_trk_id, u8 *prev_loc, u8 num_touches);
static int  ft5x06_next_avail_inlist(u16 cur_trk[], u8 *new_loc, u8 num_touches);

static int __devinit ft5x06_ts_probe(struct i2c_client *client, const struct i2c_device_id *id);
static int __devexit ft5x06_ts_remove(struct i2c_client *client);

static int  ft5x06_resume(struct i2c_client *client);
static int  ft5x06_suspend(struct i2c_client *client, pm_message_t message);

static void ft5x06_early_suspend(struct early_suspend *handler);
static void ft5x06_late_resume(struct early_suspend *handler);

extern void register_ft_i2c_adapter(struct i2c_adapter *adapter);
extern void unregister_ft_i2c_adapter(struct i2c_adapter *adapter);
/*****************************************************************************
 * Global Variables
 ****************************************************************************/

static struct workqueue_struct *ft5x06_ts_wq;

static struct input_dev * inpt = NULL;

static struct ft5x06_xydata_t g_xy_data;

static const struct i2c_device_id ft5x06_ts_id[] = {
    { FT_I2C_NAME, 0 },  { }
};

MODULE_DEVICE_TABLE(i2c, ft5x06_ts_id);

static struct i2c_driver ft5x06_ts_driver = {
    .driver = {
        .name = FT_I2C_NAME,
        .owner = THIS_MODULE,
    },
    .probe = ft5x06_ts_probe,
    .remove = __devexit_p(ft5x06_ts_remove),
#ifdef CONFIG_HAS_EARLYSUSPEND
    .suspend = NULL,
    .resume = NULL,
#else  /* CONFIG_HAS_EARLYSUSPEND */
    .suspend = ft5x06_suspend,
    .resume = ft5x06_resume,
#endif /* CONFIG_HAS_EARLYSUSPEND */
    .id_table = ft5x06_ts_id,
};
static struct i2c_driver ft5x06_ts_driver_1050 = {
    .driver = {
        .name = FT_I2C_NAME,
        .owner = THIS_MODULE,
    },
    .probe = ft5x06_ts_probe,
    .remove = __devexit_p(ft5x06_ts_remove),
    .suspend = NULL,
    .resume = NULL,
    .id_table = ft5x06_ts_id,
};


static ssize_t ft5x06_irq_status(struct device *dev, struct device_attribute *attr, char *buf)
{
    struct i2c_client *client = container_of(dev, struct i2c_client, dev);
    struct ft5x06 *ts = i2c_get_clientdata(client);

    return sprintf(buf, "%u\n", atomic_read(&ts->irq_enabled));
}


static ssize_t ft5x06_irq_enable(struct device *dev, struct device_attribute *attr, const char *buf, size_t size)
{
    struct i2c_client *client = container_of(dev, struct i2c_client, dev);
    struct ft5x06 *ts = i2c_get_clientdata(client);

    int err = 0;
    unsigned long value;


    if (size > 2)
    {
        return -EINVAL;
    }

    err = strict_strtoul(buf, 10, &value);
    if (err != 0)
    {
        return err;
    }

    switch (value)
    {
        case 0:
            if (atomic_cmpxchg(&ts->irq_enabled, 1, 0))
            {
                printk(KERN_INFO "%s() - Touch Panel IRQ %u has been DISABLED.\n", __FUNCTION__, ts->client->irq);
                disable_irq(ts->client->irq);
            }
            err = size;
            break;

        case 1:
            if (!atomic_cmpxchg(&ts->irq_enabled, 0, 1))
            {
                printk(KERN_INFO "%s() - Touch Panel IRQ %u has been ENABLED.\n", __FUNCTION__, ts->client->irq);
                enable_irq(ts->client->irq);
            }
            err = size;
            break;

        default:
            printk(KERN_ERR "%s() - Invalid input specified (%lu). Touch Panel IRQ %u --> irq_enabled = %d\n", __FUNCTION__, value, ts->client->irq, atomic_read(&ts->irq_enabled));
            err = -EINVAL;
            break;
    }

    return err;
}

static DEVICE_ATTR(irq_enable, 0777, ft5x06_irq_status, ft5x06_irq_enable);

//#ifdef CONFIG_TSW1050

int getVirtualButton(struct ft5x06 *ts, u16 y)
{
	int virt_key; // Index of the virtual keys we've defined
	int vButton = -1; // The virtual key we think has activity.

	for(virt_key=0; virt_key < FT5x06_NUM_VIRT_BUTTON; ++virt_key) {
    	// Calculate edges around button where it is still considered the button
    	if( (y > ts->platform_data->virt_btn[virt_key].centery - ts->platform_data->virt_btn_radius) &&
            (y < ts->platform_data->virt_btn[virt_key].centery + ts->platform_data->virt_btn_radius)) {
            vButton = virt_key;
            break;
        }
    }
    return vButton;
}

int processVirtualButton( struct ft5x06 *ts, u16 x, u16 y, int pressed )
{
    int vButton = getVirtualButton(ts, y);
    if(vButton < 0) {
    	// No virtual button in this location
    	return 0;
    }
	//printk(KERN_INFO "%s state=%i x=%i y=%i vButton=%i\n", __FUNCTION__, pressed, x, y, vButton);

if (g_virt_dbg_keys){
    // Send the buttons up to android only if DEBUG keys are set through /proc. otherwise no keys go to Android.
	input_report_key(ts->keyinput, ts->platform_data->virt_btn[vButton].keycode, !!pressed);
}
else
{
	//UIHAL still needs to get a key event in order to read the proc
	input_report_key(ts->keyinput, KEY_F14, !!pressed);
}

    mutex_lock(keyStateMutex);  //mutex lock for gVbKeyStates

    // Now send the buttons to the uiHAL.
    if (!pressed) {
		gVbKeyStates &= ~( 0x0001 << ( ( 0 * USED_KEY_ROWS ) + (vButton ) ) );
    } else {
		gVbKeyStates |= ( 0x0001 << ( ( 0 * USED_KEY_ROWS ) + ( vButton ) ) );
    }

	mutex_unlock(keyStateMutex);

    input_sync(ts->keyinput);

    return 0;
}

//TODO: Remove once we're sure we don't need to replicate the behavior.
// This function outputs points to Android Layer
// state: pen up =0/ pen down=1
int outputPoint(struct ft5x06 *ts, u16 x, u16 y, int state, bool id)
{
	//printk(KERN_INFO "%s x=%4i y=%4i state=%4s id=%i\n", __FUNCTION__, x, y, (state == 1) ? " TCH" : "NTCH", id);
        //ts->act_trk[touchId] = FT_TCH;
        ts->act_trk[id] = state;  //touch state  1=touch; 0=lift off

        // only send up changed x/y's ; sending all causes touch screen problem
        if (!(ts->prv_mt_pos[id][0] == x && ts->prv_mt_pos[id][1] == y) || (ts->prv_mt_tch[id] != state))
        {

	input_report_abs(ts->input, ABS_MT_TRACKING_ID, id);
        //input_report_abs(ts->input, ABS_MT_TOUCH_MAJOR, 0xe);
   	//input_report_abs(ts->input, ABS_MT_WIDTH_MAJOR, 10);
   	input_report_abs(ts->input, ABS_MT_PRESSURE, 1);
   	input_report_abs(ts->input, BTN_TOUCH, state);
   	input_report_abs(ts->input, ABS_MT_POSITION_X,  x);
   	input_report_abs(ts->input, ABS_MT_POSITION_Y,  y);

	     // Populate Crestron uiHal
 	     uiTouchData[id].touchId=id;
 	     uiTouchData[id].pressState=state;
	     uiTouchData[id].xPosition=x;
	     uiTouchData[id].yPosition=y;


	     //Send data to Android Layer
             input_mt_sync(ts->input);
             input_sync(ts->input);    //Flush input buffer

	    // if pen-up need to send again (not sure why, but ot works)
	    if (state==0) 
	     {	
		//Re-Flush input buffer on pen-up
		input_mt_sync(ts->input);
             	input_sync(ts->input);  
	    }	
        }
        ts->prv_mt_pos[id][0] = x;
        ts->prv_mt_pos[id][1] = y; 
        ts->prv_mt_tch[id] = state; 
        return 0;
}

int outputPoint750(struct ft5x06 *ts, int touchId, int state, u16 x, u16 y)
{
	//printk(KERN_INFO "%s x=%4i y=%4i prev_state=%i state=%i id=%i\n", __FUNCTION__, x, y, prev_state, state, id);
	FT5x06_DEBUG(printk(KERN_INFO "%s id=%i st=%i x=%4i y=%4i", __FUNCTION__, touchId, state, x, y);)

	input_report_abs(ts->input, ABS_MT_TRACKING_ID, touchId);
	// Sending touch_major and width_major as touch per original driver.
	input_report_abs(ts->input, ABS_MT_TOUCH_MAJOR, (state == 1) ? 0xe : FT_NTCH);
	input_report_abs(ts->input, ABS_MT_WIDTH_MAJOR, FT_SMALL_TOOL_WIDTH);

	// Previous logic to send pressure
	//	input_report_abs(ts->input, ABS_MT_PRESSURE, 1);
	//	input_report_abs(ts->input, BTN_TOUCH, state);

	input_report_abs(ts->input, ABS_MT_POSITION_X,  x);
	input_report_abs(ts->input, ABS_MT_POSITION_Y,  y);

	// Populate Crestron uiHal
	uiTouchData[touchId].touchId=touchId;
	uiTouchData[touchId].pressState=state;
	uiTouchData[touchId].xPosition=x;
	uiTouchData[touchId].yPosition=y;


	// Sync touch data.
	input_mt_sync(ts->input);
	return 0;
}

/**
 * Test to see if the touch is in the virtual keyspace.
 */
bool isInVirtKeySpace(struct ft5x06 *ts, u16 X)
{
	return (ts->platform_data->has_virt_btn &&
			(X > ts->platform_data->virt_btn_startx));
}

/**
 * Extracted from Read_xxx_1050 function above and repurposed for other touch screens.
 * Outputs points and key-up/key-down events to Android if not in the virtual
 * key area.
 */
void processTouchData(struct ft5x06 *ts, int touchId, u8 eventId, u16 X, u16 Y)
{
	//printk(KERN_ERR "b0=%d B1=%d B2=%d B3=%d B4=%d B5=%d B6=%d B7=%d\n",ReadBuffer[0],ReadBuffer[1],ReadBuffer[2],ReadBuffer[3],ReadBuffer[4],ReadBuffer[5],ReadBuffer[6],ReadBuffer[7]);
	//printk(KERN_INFO "%s touchId=%i event=%i X=%i Y=%i\n", __FUNCTION__, touchId, eventId, X, Y);

	// It used to be that we only sent changed touch data for the entire
	//touchspace. It turns out that Android expects constant updates even if
	//nothing changes. On the other hand, this doesn't make sense for the
	//virtual keys.
	if(eventId == 1) { // Touch
		if(isInVirtKeySpace(ts, X)) {
			if(ts->prv_mt_pos[touchId][0] == X && ts->prv_mt_pos[touchId][1] == Y) {
				// Only check to make sure we don't resend events for the virtual
				//keys. No change, don't update the touch.
				return;
			}
			// Send virtual key down
			processVirtualButton(ts, X, Y, FT_TCH);
			ts->act_trk[touchId] = FT_TCH;
			ts->prv_mt_pos[touchId][0] = X;
			ts->prv_mt_pos[touchId][1] = Y;
			return;
		}
		// New touch, or new touch location
		outputPoint750(ts, touchId, FT_TCH, X, Y);
		ts->act_trk[touchId] = FT_TCH;
		ts->prv_mt_pos[touchId][0] = X;
		ts->prv_mt_pos[touchId][1] = Y;
	}
	else { // Lift off
		if(ts->act_trk[touchId] == FT_NTCH) {
			// If there was previously no touch, don't report again.
			return;
		}
		if(isInVirtKeySpace(ts, X)) {
			// Send virtual key up
			processVirtualButton(ts, X, Y, FT_NTCH);
			ts->act_trk[touchId] = FT_NTCH;
			ts->prv_mt_pos[touchId][0] = 0;
			ts->prv_mt_pos[touchId][1] = 0;
			return;
		}
		// Previous touch released.
		outputPoint750(ts, touchId, FT_NTCH, X, Y);
		ts->act_trk[touchId] = FT_NTCH;
		ts->prv_mt_pos[touchId][0] = 0;
		ts->prv_mt_pos[touchId][1] = 0;
	}
}

/**
 * Despite its name, reads touch data for all touchpanels other than the 1050.
 */
void read_touchdata_750(struct ft5x06 *ts)
{
    int retval = 0;
    ft506_data tch_data; // Data to be read from the device
    // Temporaries used just to send to processTouchData
    u16 x = 0;			// Read coordinates into these variables
    u16 y = 0;
    u8 touchId = 0;		// The touchId that we eventually
    u8 event = 0;		// The event, touchdown or touchup
    u8 numTouches = 0;
    u32 idProcessed = 0;

	const int FT_NULL_TOUCH = 4095; // We send the null touch as a lift-up event.
    u8 iTouch;

    retval = i2c_smbus_read_i2c_block_data(ts->client, FT5x06_WMREG_DEVICE_MODE, sizeof(ft506_data), (u8 *)&tch_data);
    if (unlikely(retval < 0))   //initialized to 0
    {
        printk(KERN_ERR "%s() - ERROR: Could not read from the Touch Panel registers.\n", __FUNCTION__);
    }
    else
    {
    	numTouches = GET_NUM_TOUCHES(tch_data.status);
    	//printk(KERN_INFO "Num touches = %i\n", numTouches);
    	idProcessed = 0;
    	/* Always call processTouchData with both touch points' data. */
    	for(iTouch=0; iTouch < numTouches; ++iTouch) {
    	//for(iTouch=0; iTouch < MAX_TOUCH_POINTS_SUPPORTED; ++iTouch) {
    		x = GET_XCOORD(tch_data.point[iTouch]);
    		y = GET_YCOORD(tch_data.point[iTouch]);
    		touchId = GET_TOUCH_ID(tch_data.point[iTouch]);
    		event = GET_EVENT(tch_data.point[iTouch]);

    		//printk(KERN_INFO "RJK: npts=%d; id1=%d; x1=%d; y1=%d;\n", tch_data.status, touchId, x1, y1);
    		// Convert the event to the values that processTouchData expects.
    		//We send the NULL_TOUCH as a TOUCH_UP event.
    		if((event == EP0xx0M06_TOUCH_DOWN || (event == EP0xx0M06_TOUCH_ON)) &&
    		    x != FT_NULL_TOUCH && y != FT_NULL_TOUCH)
    		{
    			event = 1; // Touch down event
    		}
    		else
    		{
    			event = 0; // Touch up event
    		}
    		idProcessed |= (1 << touchId);
    		processTouchData(ts, touchId, event, x, y);
    	}
    	//printk(KERN_INFO "%x processed\n", idProcessed);
		/* Process touchIds that aren't in the list of current touch points. */
		for(iTouch=0; iTouch < MAX_TOUCH_POINTS_SUPPORTED; ++iTouch) {
			if((idProcessed & (1 << iTouch)) == 0) {
	    		idProcessed |= (1 << iTouch);
				//processTouchData(ts, &tch_data, iTouch, 0, 0, 0);
	    		// Send the release with the previous touch's location
	    		processTouchData(ts, iTouch, 0, ts->prv_mt_pos[iTouch][0], ts->prv_mt_pos[iTouch][1]);
			}
		}
    }
}

void EP0xxM06_Read_TouchData_1050(struct ft5x06 *ts)
{
	u32 bytesRead=0;
	u8 ReadBuffer[10];
	u8 eventId = 0;	// The event ID of a particular touch-point
	int touchId = 0;// The touch ID of a particular touch-point
	u8 packetId = 0;// The packet ID of I2C read packet
	u8 isValid = 0;	// The packet ID of I2C read packet

	u16 X=0, Y=0, Z=0;  // bit shifted x and y

	int loopCnt=0;
    u32 idProcessed = 0;
    u32 touchBit = 0;

    FT5x06_DEBUG( \
    	struct timeval tstart; \
		struct timeval tend; \
		do_gettimeofday(&tstart); \
    )

	do
	{
		// Clear local data-transfer buffer.
		memset(&ReadBuffer, 0, sizeof(ReadBuffer));
		// Only read if the interrupt line is being held low
		if(ts->platform_data->hw_status()) {
			printk(KERN_INFO "Interrupt line is not low\n");
			continue;
		}
		FT5x06_DEBUG( \
			do_gettimeofday(&tend);
			printk(KERN_INFO "%i us\n", (tend.tv_usec - tstart.tv_usec)); \
			tstart = tend; \
		)
		// Per spec, needs at least 60 microsecond delay between reads
		//However, in practice it does not seem to need it. Processing consumes
		//enough time except when the diagnostic packets are involved.
		bytesRead=i2c_master_recv(ts->client, (u8 *)&ReadBuffer, 10);
		if(bytesRead < 10) {
			printk(KERN_ERR "1050: Only read %i bytes.\n", bytesRead);
			continue;
		}

		eventId = GET_EVENT_1050(ReadBuffer);	  // B0=1: Touch Down; B0=0: Lift Off
		touchId = GET_TOUCH_ID_1050(ReadBuffer);  // id=0 or id=1  max 2 touches
		packetId= GET_PACKET_ID_1050(ReadBuffer); // 0x04 cmd or 0x03 diag
		isValid = GET_PACKET_VALID_1050(ReadBuffer);// is read valid

		FT5x06_DEBUG(printk(KERN_ERR "b0=%d B1=%d B2=%d B3=%d B4=%d B5=%d B6=%d B7=%d\n",ReadBuffer[0],ReadBuffer[1],ReadBuffer[2],ReadBuffer[3],ReadBuffer[4],ReadBuffer[5],ReadBuffer[6],ReadBuffer[7]);)

		// We only report touch data when we've gotten a packet with the same
		//touchId a second time *since* we last sent data up. Presumably, this
		//means we will report data for all simultaneous touches.
		touchBit = (1 << touchId);
		if(idProcessed & touchBit) {
			// Already processed this touch in this loop before, send the
			//input_sync to communicate it up before proceeding.
			input_sync(ts->input);
			idProcessed = 0;
		}
		idProcessed |= touchBit;

		if (isValid && packetId == 0x04)
		{
			//Parse X Y Z coords;  nothing done with Z yet.
			X = GET_XCOORD_1050(ReadBuffer);
			Y = GET_YCOORD_1050(ReadBuffer);
			//Z = GET_ZCOORD_1050(ReadBuffer);
			FT5x06_DEBUG(printk(KERN_INFO "tch=%i ev=%i X=%4i Y=%4i\n", touchId, eventId, X, Y);)

			//TODO: Need more work before we turn this back on. If we skip
			//a packet, we have to delay more because we'll do our next read in
			//under 60 microseconds.
//			// Filter packets that don't have "significant" changes. Note that
//			//processTouchData() also filters non-changing packets in the virtual
//			//keyspace.
//			++loopCnt;
//			if(loopCnt < MAX_NUM_PACKETS_READ_1050 &&
//			   ts->prv_mt_tch[touchId] == eventId &&
//			   !isInVirtKeySpace(ts, X)) {
//				// Drop data, need to throttle data going to Android
//				loopCnt = 0;
//				break;
//			}
			//process data
			processTouchData(ts, touchId, eventId, X, Y);
		}
		else if(packetId == 0x03)
		{
			printk(KERN_INFO "%s: Diagnostic packet ID = 0x%x.\r\n", __FUNCTION__, packetId);
			udelay(PACKET_READ_DELAY_1050);
		}
		else
		{
			printk(KERN_INFO "%s: Unrecognized touch packet ID = 0x%x.\r\n", __FUNCTION__, packetId);
			udelay(PACKET_READ_DELAY_1050);
		}
	} while(!ts->platform_data->hw_status());
	// Pickup any stragglers. idProcessed above handles sending the input_sync
	//while looping, this is for any remaining. This is necessary to make sure
	//the last key event is picked up before exiting.
	input_sync(ts->input);
}

/* The ft5x06_xy_worker function reads the XY coordinates and sends them to
 * the input layer.  It is scheduled from the interrupt (or timer).
 */
void ft5x06_xy_worker(struct work_struct *work)
{
    struct ft5x06 *ts = container_of(work, struct ft5x06, work);
    int retval = 0;
    ft506_data tch_data;

    u8 id;
    u16 cur_trk[FT_NUM_TRK_ID];

    if (inpt == NULL )
    {
        inpt = ts->input;
    }

    g_xy_data.gest_id = 0;

    if(gProductId == TSW1050)
    {
        EP0xxM06_Read_TouchData_1050(ts);
    }
    else
    {
    	read_touchdata_750(ts);
    }

    /* The 1050 logic sends the touch information up on its own. */
    if(gProductId == TSW750 || gProductId == TST600 || gProductId == TST900 || gProductId == TSW550 || gProductId == TSM750)
    {
        /* signal the view motion event */
        input_sync(ts->input);
    }

//    if(gProductId == TSW1050) {
//		for (id = 0; id < FT_NUM_TRK_ID; id++)
//		{
//			/* update platform data for the current MT information */
//			ts->act_trk[id] = cur_trk[id];
//		}
//    }

exit_xy_worker:
    if (ts->client->irq != 0)
    {

        if(gProductId == TSW750 || gProductId == TST600 || gProductId == TST900 || gProductId == TSW550 || gProductId == TSM750)
        {
            msleep(FT5x06_SCAN_DELAY);

            /* re-enable the interrupt after processing */
            enable_irq(ts->client->irq);
        }
        else
        {
            msleep(FT5x06_SCAN_DELAY);
        }

    }
    return;
}


static int ft5x06_inlist(u16 prev_track[], u8 cur_trk_id, u8 *prev_loc, u8 num_touches)
{
    u8 id = 0;


    *prev_loc = FT_IGNR_TCH;

    for (id = 0, *prev_loc = FT_IGNR_TCH; (id < num_touches); id++)
    {
        if (prev_track[id] == cur_trk_id)
        {
            *prev_loc = id;
            break;
        }
    }

    return ((*prev_loc < FT_NUM_TRK_ID) ? true : false);
}


static int ft5x06_next_avail_inlist(u16 cur_trk[], u8 *new_loc, u8 num_touches)
{
    u8 id;

    for (id = 0, *new_loc = FT_IGNR_TCH; (id < num_touches); id++)
    {
        if (cur_trk[id] > FT_NUM_TRK_ID)
        {
            *new_loc = id;
            break;
        }
    }

    return ((*new_loc < FT_NUM_TRK_ID) ? true : false);
}


/*************************************************************************
 * ISR function. This function is general, initialized in drivers init
 * function
 ************************************************************************/
static irqreturn_t ft5x06_irq(int irq, void *handle)
{
    struct ft5x06 *ts = (struct ft5x06 *) handle;

if (gProductId == TST600 || gProductId == TST900 || gProductId == TSW750 || gProductId == TSW550 || gProductId == TSM750)
{
    /* disable further interrupts until this interrupt is processed */
    disable_irq_nosync(ts->client->irq);
}

    /* schedule motion signal handling */
    queue_work(ft5x06_ts_wq, &ts->work);
    return IRQ_HANDLED;
}


/*************************************************************************
 * GPIO Helper Functions
 ************************************************************************/

static void ft5x06_reset_panel_via_gpio(int reset_gpio)
{
#if FT5x06_DEBUG_VERBOSE
    printk(KERN_DEBUG "%s() - Toggling GPIO %d to reset the Touch Panel...\n", __FUNCTION__, reset_gpio);
#endif /* FT5x06_DEBUG_VERBOSE */

    gpio_direction_output(reset_gpio, 1);
    msleep(20);
    gpio_direction_output(reset_gpio, 0);
    msleep(20);
    gpio_direction_output(reset_gpio, 1);
    msleep(300);
}


static bool ft5x06_poll_gpio(const int gpio_num, const int requested_gpio_val)
{
    const int poll_count_limit = 20;
    const int poll_delay_ms    = 5;
    int poll_count = 0;
    int gpio_val   = -1;


#if FT5x06_DEBUG_VERBOSE
    printk(KERN_DEBUG "%s() - Waiting for GPIO %d to go to %d...\n", __FUNCTION__, gpio_num, requested_gpio_val);
#endif /* FT5x06_DEBUG_VERBOSE */

    while ((poll_count_limit > poll_count++) && (requested_gpio_val != gpio_val))
    {
        msleep(poll_delay_ms);
        gpio_val = gpio_get_value(gpio_num);
    }

    return (requested_gpio_val == gpio_val);
}

/* Set the TP controller to Firmware update mode return 0 for success
 * caller needs to reset TP for exit update mode */
static int ft5x06_enter_fwupdate_mode(struct ft5x06 *ts)
{
	int  retval = 0;
	int  i      = 0;
	u8  temp_buffer[4];

    printk(KERN_INFO "%s() - Step 1: Reset the CTPM\n", __FUNCTION__);
    retval = i2c_smbus_write_byte_data(ts->client, FT5x06_WMREG_RESET, FT5x06_CMD_RESET_CTPM_H);
    if (0 > retval) {
        printk(KERN_ERR "%s() - ERROR: Could not write the CTPM Reset command (high byte) to the CTPM.\n", __FUNCTION__);
        goto error_return;
    }

    msleep(50);

    retval = i2c_smbus_write_byte_data(ts->client, FT5x06_WMREG_RESET, FT5x06_CMD_RESET_CTPM_L);
    if (0 > retval) {
        printk(KERN_ERR "%s() - ERROR: Could not write the CTPM Reset command (low byte) to the CTPM.\n", __FUNCTION__);
        goto error_return;
    }

    msleep(30);

    printk(KERN_INFO "%s() - Step 2: Put the CTPM in Firmware Upgrade mode\n", __FUNCTION__);
    temp_buffer[0] = 0x55;
    temp_buffer[1] = 0xAA;

    for (i = 0; ((i < 5) && (0 >= retval)); i++) {
        struct i2c_msg msg =    {
                                    .addr   = ts->client->addr,
                                    .flags  = 0,
                                    .len    = 2,
                                    .buf    = temp_buffer,
                                };
        msleep(5);

        retval = i2c_transfer(ts->client->adapter, &msg, 1);
    }

    if (0 >= retval) {
        printk(KERN_ERR "%s() - ERROR: Could not put the CTPM in Firmware Update mode.\n", __FUNCTION__);
        goto error_return;
    }
    return 0;

error_return:
    return -1;
}

/*************************************************************************
 * Factory Mode Helper Functions
 ************************************************************************/

static int ft5x06_enter_factory_mode(struct ft5x06 * ts)
{
    int retval = 0;
    u8  regval = 0;


    if (ts->client->irq)
    {
#if FT5x06_DEBUG_VERBOSE
        printk(KERN_DEBUG "%s() - Disabling Touch Panel interrupts...\n", __FUNCTION__);
#endif /* FT5x06_DEBUG_VERBOSE */
        disable_irq(ts->client->irq);
    }

#if FT5x06_DEBUG_VERBOSE
    printk(KERN_DEBUG "%s() - Putting the Touch Panel in Factory Mode...\n", __FUNCTION__);
#endif /* FT5x06_DEBUG_VERBOSE */

    regval = FT5x06_MODE_FACTORY;
    retval = i2c_smbus_write_i2c_block_data(ts->client, FT5x06_WMREG_DEVICE_MODE, sizeof(u8), &regval);
    if (0 != retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not write to the Device Mode register.\n", __FUNCTION__);
        goto error_enable_irq;
    }

    msleep(100);

#if FT5x06_DEBUG_VERBOSE
    printk(KERN_DEBUG "%s() - Verifying that the Touch Panel is in Factory Mode...\n", __FUNCTION__);
#endif /* FT5x06_DEBUG_VERBOSE */

    retval = i2c_smbus_read_i2c_block_data(ts->client, FT5x06_FMREG_DEVICE_MODE, sizeof(u8), &regval);
    if (0 > retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not read from the Device Mode register.\n", __FUNCTION__);
        goto error_enable_irq;
    }

    if ((regval & FT5x06_MODE_MASK) != FT5x06_MODE_FACTORY)
    {
        printk(KERN_ERR "%s() - ERROR: The Touch Panel was not put in Factory Mode. The Device Mode register contains 0x%02X\n", __FUNCTION__, regval);
        retval = FT5x06_ERR_NOT_FACTORY_MODE;
        goto error_enable_irq;
    }

    return 0;

error_enable_irq:
    if (ts->client->irq)
    {
#if FT5x06_DEBUG_VERBOSE
        printk(KERN_DEBUG "%s() - Enabling Touch Panel interrupts.\n", __FUNCTION__);
#endif /* FT5x06_DEBUG_VERBOSE */
        enable_irq(ts->client->irq);
    }

    return retval;
}


static int ft5x06_exit_factory_mode(struct ft5x06 * ts)
{
    int retval = 0;
    u8  regval = 0;


#if FT5x06_DEBUG_VERBOSE
    printk(KERN_DEBUG "%s() - Putting the Touch Panel in Working Mode...\n", __FUNCTION__);
#endif /* FT5x06_DEBUG_VERBOSE */

    regval = FT5x06_MODE_WORKING;
    retval = i2c_smbus_write_i2c_block_data(ts->client, FT5x06_WMREG_DEVICE_MODE, sizeof(u8), &regval);
    if (0 != retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not write to the Device Mode register.\n", __FUNCTION__);
        goto error_enable_irq;
    }

    msleep(100);

#if FT5x06_DEBUG_VERBOSE
    printk(KERN_DEBUG "%s() - Verifying that the Touch Panel is in Working Mode...\n", __FUNCTION__);
#endif /* FT5x06_DEBUG_VERBOSE */

    retval = i2c_smbus_read_i2c_block_data(ts->client, FT5x06_FMREG_DEVICE_MODE, sizeof(u8), &regval);
    if (0 > retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not read from the Device Mode register.\n", __FUNCTION__);
        goto error_enable_irq;
    }

    if ((regval & FT5x06_MODE_MASK) != FT5x06_MODE_WORKING)
    {
        printk(KERN_ERR "%s() - ERROR: The Touch Panel was not put in Working Mode. The Device Mode Register contains 0x%02X\n", __FUNCTION__, regval);
        retval = FT5x06_ERR_NOT_WORKING_MODE;
        goto error_enable_irq;
    }

    retval = 0;

error_enable_irq:
    if (ts->client->irq)
    {
#if FT5x06_DEBUG_VERBOSE
        printk(KERN_DEBUG "%s() - Enabling Touch Panel interrupts.\n", __FUNCTION__);
#endif /* FT5x06_DEBUG_VERBOSE */
        enable_irq(ts->client->irq);
    }

    return retval;
}


static int ft5x06_read_data(struct i2c_client * client, char * output_buffer, ssize_t output_buffer_size, ssize_t * p_num_read_chars)
{
    int retval  = 0;
    int i       = 0;
    u16 dataval = 0x0000;
    u8  devmode = 0x00;
    u8  rownum  = 0x00;

    u8 read_buffer[FT5x06_NUM_RX * 2];


#if FT5x06_DEBUG_VERBOSE
    printk(KERN_DEBUG "%s() - Verifying that the Touch Panel is in Factory Mode...\n", __FUNCTION__);
#endif /* FT5x06_DEBUG_VERBOSE */

    retval = i2c_smbus_read_i2c_block_data(client, FT5x06_FMREG_DEVICE_MODE, sizeof(u8), &devmode);
    if (0 > retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not read from the Device Mode register.\n", __FUNCTION__);
        goto error_return;
    }

    if (FT5x06_MODE_FACTORY != (devmode & FT5x06_MODE_MASK))
    {
        printk(KERN_ERR "%s() - ERROR: The Touch Panel is not in Factory Mode.\n", __FUNCTION__);
        retval = FT5x06_ERR_NOT_FACTORY_MODE;
        goto error_return;
    }

#if FT5x06_DEBUG_VERBOSE
    printk(KERN_DEBUG "%s() - Initiating a scan for raw data...\n", __FUNCTION__);
#endif /* FT5x06_DEBUG_VERBOSE */

    devmode |=  FT5x06_SCAN_START;
    retval = i2c_smbus_write_i2c_block_data(client, FT5x06_FMREG_DEVICE_MODE, sizeof(u8), &devmode);
    if (0 != retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not initiate a raw data scan.\n", __FUNCTION__);
        goto error_return;
    }

    msleep(FT5x06_SCAN_DELAY);

    retval = i2c_smbus_read_i2c_block_data(client, FT5x06_FMREG_DEVICE_MODE, sizeof(u8), &devmode);
    if (0 > retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not read the updated value of the Device Mode register.\n", __FUNCTION__);
        goto error_return;
    }

    if (FT5x06_SCAN_DONE != (devmode & FT5x06_SCAN_MASK))
    {
        printk(KERN_ERR "%s() - ERROR: The raw data scan did not complete after %u ms.\n", __FUNCTION__, FT5x06_SCAN_DELAY);
        retval = FT5x06_ERR_SCAN_NOT_DONE;
        goto error_return;
    }

#if FT5x06_DEBUG_VERBOSE
    printk(KERN_DEBUG "%s() - Reading raw data...\n", __FUNCTION__);
#endif /* FT5x06_DEBUG_VERBOSE */

    for (rownum = 0; rownum < FT5x06_NUM_TX; rownum ++)
    {
        memset(read_buffer, 0x00, (FT5x06_NUM_RX * 2));

        retval = i2c_smbus_write_i2c_block_data(client, FT5x06_FMREG_ROW_ADDR, sizeof(u8), &rownum);
        if (0 != retval)
        {
            printk(KERN_ERR "%s() - ERROR: Could not write the row number.\n", __FUNCTION__);
            goto error_return;
        }

        msleep(1);

        /* Read the data for this row */
        retval = i2c_smbus_read_i2c_block_data(client, FT5x06_FMREG_RAWDATA_0_H, (FT5x06_NUM_RX * 2), read_buffer);
        if (0 > retval)
        {
            printk(KERN_ERR "%s() - ERROR: Could not read row %u raw data.\n", __FUNCTION__, rownum);
            goto error_return;
        }

        /* Each data value can be up to 5 chars long. Add a space and we need 6 chars to store it. */
        for (i = 0; ((i < retval) && ((*p_num_read_chars + 6) < output_buffer_size)); i += 2)
        {
            dataval  = read_buffer[i];
            dataval  = (dataval << 8);
            dataval |= read_buffer[i+1];

            *p_num_read_chars += sprintf(&(output_buffer[*p_num_read_chars]), "%u ", dataval);
        }

        output_buffer[*p_num_read_chars-1] = '\n';
    }

    retval = 0;

error_return:
    return retval;
}

/*************************************************************************
 * Firmware Update
 ************************************************************************/

static int ft5x06_perform_fw_upgrade(struct device *dev, const u8 * firmware_src_buffer, u32 firmware_size)
{
    struct i2c_client *client = container_of(dev, struct i2c_client, dev);
    struct ft5x06     *ts     = (struct ft5x06 *)i2c_get_clientdata(client);

    int  retval = 0;
    int  i      = 0;
    int  j      = 0;

    u32 num_packets = 0;
    u16 temp_val    = 0;
    u8  checksum    = 0;

    u8  temp_buffer[4];
    u8  packet_buffer[FT5x06_PACKET_LENGTH + 6];


    if (NULL == firmware_src_buffer)
    {
        printk(KERN_ERR "%s() - ERROR: Firmware Source Buffer pointer is NULL.\n", __FUNCTION__);
        retval = -EINVAL;
        goto error_return;
    }

    if (0 == firmware_size)
    {
        printk(KERN_ERR "%s() - ERROR: Firmware Source Size is ZERO.\n", __FUNCTION__);
        retval = -EINVAL;
        goto error_return;
    }

    /*********************************************************************************************/
    printk(KERN_INFO "%s() - Step 1: Reset the CTPM\n", __FUNCTION__);

	retval = ft5x06_enter_fwupdate_mode(ts);
	if (0 != retval) {
		goto error_reset;
	}
    /*********************************************************************************************/
    printk(KERN_INFO "%s() - Step 3: Read the CTPM ID\n", __FUNCTION__);

    temp_buffer[0] = FT5x06_CMD_GET_ID;
    temp_buffer[1] = FT5x06_CMD_GET_ID_P1;
    temp_buffer[2] = FT5x06_CMD_GET_ID_P2;
    temp_buffer[3] = FT5x06_CMD_GET_ID_P3;

    retval = i2c_master_send(ts->client, temp_buffer, 4);
    if (0 > retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not write the Get ID command to the CTPM.\n", __FUNCTION__);
        goto error_reset;
    }

    msleep(10);

    retval = i2c_master_recv(ts->client,temp_buffer, 2);
    if (0 > retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not read the ID from the CTPM.\n", __FUNCTION__);
        goto error_reset;
    }

    if ((FT5x06_CPTM_ID_COMPANY != temp_buffer[0]) || (FT5x06_CPTM_ID_PRODUCT != temp_buffer[1]))
    {
        printk(KERN_ERR "%s() - ERROR: Invalid CPTM ID. Expected 0x%02X%02X, got 0x%02X%02X.\n", __FUNCTION__, FT5x06_CPTM_ID_COMPANY, FT5x06_CPTM_ID_PRODUCT, temp_buffer[0], temp_buffer[1]);
        retval = FT5x06_ERR_INVALID_ID;
        goto error_reset;
    }

    /*********************************************************************************************/
    printk(KERN_INFO "%s() - Step 4: Erase the old CTPM firmware\n", __FUNCTION__);

    temp_buffer[0] = FT5x06_CMD_ERASE_FW;

    retval = i2c_master_send(ts->client, temp_buffer, 1);
    if (0 > retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not write the Erase Firmware command to the CTPM.\n", __FUNCTION__);
        goto error_reset;
    }

    msleep(1500);

    /*********************************************************************************************/
    printk(KERN_INFO "%s() - Step 5: Write the new CTPM firmware to CTPM flash\n", __FUNCTION__);

    /* We write everything but the last 8 bytes in packets.
     * The first 6 of the last 8 bytes will be written in the footer.
     * The final 2 bytes (which seem to be always 0xFF and 0x00) don't get written.
     */
    firmware_size = firmware_size - 8;
    num_packets   = (firmware_size) / FT5x06_PACKET_LENGTH;

    packet_buffer[0] = 0xBF;
    packet_buffer[1] = 0x00;

    /* Write whole packets */
    for (i = 0; i < num_packets; i++)
    {
        /* Target offset */
        temp_val = i * FT5x06_PACKET_LENGTH;

        packet_buffer[2] = (u8)(0x00FF & (temp_val >> 8));
        packet_buffer[3] = (u8)(0x00FF & (temp_val));

        /* Num bytes following header */
        temp_val = FT5x06_PACKET_LENGTH;

        packet_buffer[4] = (u8)(0x00FF & (temp_val >> 8));
        packet_buffer[5] = (u8)(0x00FF & (temp_val));

        for (j = 0; j < FT5x06_PACKET_LENGTH; j++)
        {
            /* Process byte j of packet i... */
            packet_buffer[6 + j] = firmware_src_buffer[(i * FT5x06_PACKET_LENGTH) + j];
            checksum ^= packet_buffer[6 + j];
        }

        retval = i2c_master_send(ts->client, packet_buffer, FT5x06_PACKET_LENGTH + 6);
        if (0 > retval)
        {
            printk(KERN_ERR "%s() - ERROR: Could not write packet %u of %u to the CTPM.\n", __FUNCTION__, i, num_packets);
            goto error_reset;
        }

        msleep(20);

        if (0 == ((i * FT5x06_PACKET_LENGTH) % 1024))
        {
            printk(KERN_DEBUG "%s() - Uploaded %6d of %6u bytes.\n", __FUNCTION__, (i * FT5x06_PACKET_LENGTH), firmware_size);
        }
    }

    printk(KERN_DEBUG "%s() - Uploaded %6d of %6u bytes.\n", __FUNCTION__, (i * FT5x06_PACKET_LENGTH), firmware_size);

    /* Write a partial packet if necessary */
    if (0 != (firmware_size % FT5x06_PACKET_LENGTH))
    {
        /* Target offset */
        temp_val = num_packets * FT5x06_PACKET_LENGTH;

        packet_buffer[2] = (u8)(0x00FF & (temp_val >> 8));
        packet_buffer[3] = (u8)(0x00FF & (temp_val));

        /* Num bytes following header */
        temp_val = (firmware_size % FT5x06_PACKET_LENGTH);

        packet_buffer[4] = (u8)(0x00FF & (temp_val >> 8));
        packet_buffer[5] = (u8)(0x00FF & (temp_val));

        for (j = 0; j < temp_val; j++)
        {
            packet_buffer[6 + j] = firmware_src_buffer[(num_packets * FT5x06_PACKET_LENGTH) + j];
            checksum ^= packet_buffer[6 + j];
        }

        retval = i2c_master_send(ts->client, packet_buffer, temp_val + 6);
        if (0 > retval)
        {
            printk(KERN_ERR "%s() - ERROR: Could not write partial packet to the CTPM.\n", __FUNCTION__);
            goto error_reset;
        }

        msleep(20);
    }

    printk(KERN_DEBUG "%s() - Uploaded %6d of %6u bytes.\n", __FUNCTION__, (i * FT5x06_PACKET_LENGTH) + temp_val, firmware_size);

    /* Write the firmware footer */
    for (i = 0; i < 6; i++)
    {
        packet_buffer[2] = 0x6F;
        packet_buffer[3] = 0xFA + i;
        packet_buffer[4] = 0x00;
        packet_buffer[5] = 0x01;

        packet_buffer[6] = firmware_src_buffer[firmware_size + i];
        checksum ^= packet_buffer[6];

        retval = i2c_master_send(ts->client, packet_buffer, 7);
        if (0 > retval)
        {
            printk(KERN_ERR "%s() - ERROR: Could not write partial packet to the CTPM.\n", __FUNCTION__);
            goto error_reset;
        }

        msleep(20);
    }

    /*********************************************************************************************/
    printk(KERN_INFO "%s() - Step 6: Checksum verification\n", __FUNCTION__);

    temp_buffer[0] = FT5x06_CMD_GET_CHECKSUM;

    retval = i2c_master_send(ts->client, temp_buffer, 1);
    if (0 > retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not write the Get Checksum command to the CTPM.\n", __FUNCTION__);
        goto error_reset;
    }

    msleep(10);

    retval = i2c_master_recv(ts->client,temp_buffer, 1);
    if (0 > retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not read the Checksum from the CTPM.\n", __FUNCTION__);
        goto error_reset;
    }

    if (checksum != temp_buffer[0])
    {
        printk(KERN_ERR "%s() - ERROR: Checksum (0x%02X) did not match calculated value (0x%02X).\n", __FUNCTION__, temp_buffer[0], checksum);
        retval = FT5x06_ERR_INVALID_CHECKSUM;
        goto error_reset;
    }

    /*********************************************************************************************/
    printk(KERN_INFO "%s() - Step 7: Reset the CTPM firmware\n", __FUNCTION__);

    temp_buffer[0] = FT5x06_CMD_RESET_FW;

    retval = i2c_master_send(ts->client, temp_buffer, 1);
    if (0 > retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not write the Reset Firmware command to the CTPM.\n", __FUNCTION__);
        goto error_reset;
    }

    retval = 0;

error_reset:
    /*********************************************************************************************/
    printk(KERN_INFO "%s() - Step 8: Reset the CTPM\n", __FUNCTION__);

    ft5x06_reset_panel_via_gpio(ts->platform_data->reset_gpio);

    printk(KERN_INFO "%s() - FIRMWARE UPDATE COMPLETE - Update %s\n", __FUNCTION__, ((0 == retval) ? "Succeeded" : "Failed"));

error_return:
    return retval;
}


/*************************************************************************
 * SYSFS Store and Show functions
 ************************************************************************/

static ssize_t ft5x06_version_show(struct device *dev, struct device_attribute *attr, char *buf)
{
    return snprintf(buf, PAGE_SIZE, "%04X\n", FT5x06_DRIVER_VERSION);
}


static ssize_t ft5x06_rawbase_show(struct device *dev, struct device_attribute *attr, char *buf)
{
    struct i2c_client *client = container_of(dev, struct i2c_client, dev);
    struct ft5x06     *ts     = (struct ft5x06 *)i2c_get_clientdata(client);

    ssize_t num_read_chars = 0;
    int     retval = 0;

    mutex_lock(&ts->device_mode_mutex);

    retval = ft5x06_enter_factory_mode(ts);
    if (0 != retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not put the Touch Panel in Factory Mode.\n", __FUNCTION__);
        goto error_restore_mode;
    }

    retval = ft5x06_read_data(ts->client, buf, PAGE_SIZE, &num_read_chars);
    if (0 != retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not read Raw Data from the Touch Panel.\n", __FUNCTION__);
        goto error_restore_mode;
    }

error_restore_mode:
    retval = ft5x06_exit_factory_mode(ts);
    if (0 != retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not return the Touch Panel to Working Mode.\n", __FUNCTION__);
        goto error_return;
    }

error_return:
    mutex_unlock(&ts->device_mode_mutex);
    return num_read_chars;
}


static ssize_t ft5x06_crosstalk_show(struct device *dev, struct device_attribute *attr, char *buf)
{
    struct i2c_client *client = container_of(dev, struct i2c_client, dev);
    struct ft5x06     *ts     = (struct ft5x06 *)i2c_get_clientdata(client);

    const u8 rx_offset = ((FT5x06_CROSSTALK_TEST_TYPE_ODD == ts->crosstalk_test_type) ? 1 : 0);
    ssize_t  num_read_chars = 0;

    u8  num_processed_rx_cac_values = 0x00;
    u8  i        = 0x00;
    u8  regval   = 0x00;
    int retval   = 0;

    u8  original_rx_cac[FT5x06_NUM_RX];

    mutex_lock(&ts->device_mode_mutex);

    memset(original_rx_cac, 0x00, FT5x06_NUM_RX);

    retval = ft5x06_enter_factory_mode(ts);
    if (0 != retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not put the device in Factory Mode.\n", __FUNCTION__);
        goto error_restore_mode;
    }

    /* Preserve the original values of the even or odd RXn_CAC registers and then set them to 0x00. */

#if FT5x06_DEBUG_VERBOSE
    printk(KERN_DEBUG "%s() - Modifying the RXn_CAC register values...\n", __FUNCTION__);
#endif /* FT5x06_DEBUG_VERBOSE */

    for (i = rx_offset; i < FT5x06_NUM_RX; i += 2)
    {
        retval = i2c_smbus_read_i2c_block_data(ts->client, (FT5x06_FMREG_RX_0_CAC + i), sizeof(u8), &regval);
        if (0 > retval)
        {
            printk(KERN_ERR "%s() - ERROR: Could not read from the RX%u_CAC register.\n", __FUNCTION__, i);
            goto error_restore_cac_values;
        }

        original_rx_cac[num_processed_rx_cac_values] = regval;
        num_processed_rx_cac_values++;

        regval = 0x00;
        retval = i2c_smbus_write_i2c_block_data(ts->client, (FT5x06_FMREG_RX_0_CAC + i), sizeof(u8), &regval);
        if (0 != retval)
        {
            printk(KERN_ERR "%s() - ERROR: Could not write 0x00 to the RX%u_CAC register.\n", __FUNCTION__, i);
            goto error_restore_cac_values;
        }
    }

    msleep(100);

    retval = ft5x06_read_data(ts->client, buf, PAGE_SIZE, &num_read_chars);
    if (0 != retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not read Raw Data from the Touch Panel.\n", __FUNCTION__);
        goto error_restore_cac_values;
    }

error_restore_cac_values:
#if FT5x06_DEBUG_VERBOSE
    printk(KERN_DEBUG "%s() - Restoring the RXn_CAC register values...\n", __FUNCTION__);
#endif /* FT5x06_DEBUG_VERBOSE */

    for (i = 0; i < num_processed_rx_cac_values; i++)
    {
        retval = i2c_smbus_write_i2c_block_data(ts->client, (FT5x06_FMREG_RX_0_CAC + (2 * i) + rx_offset), sizeof(u8), &(original_rx_cac[i]) );
        if (0 != retval)
        {
            printk(KERN_ERR "%s() - ERROR: Could not restore the original value of the RX%u_CAC register.\n", __FUNCTION__, (2 * i) + rx_offset);
            goto error_restore_mode;
        }
    }

error_restore_mode:
    retval = ft5x06_exit_factory_mode(ts);
    if (0 != retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not return the Touch Panel to Working Mode.\n", __FUNCTION__);
    }

    mutex_unlock(&ts->device_mode_mutex);
    return num_read_chars;
}


static ssize_t ft5x06_crosstalk_store(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
    struct i2c_client *client = container_of(dev, struct i2c_client, dev);
    struct ft5x06     *ts     = (struct ft5x06 *)i2c_get_clientdata(client);
    unsigned long value = 0;
    int retval = 0;

    mutex_lock(&ts->device_mode_mutex);

    retval = strict_strtoul(buf, 10, &value);
    if (0 != retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not convert the given input to a number. The given input was: \"%s\"\n", __FUNCTION__, buf);
        goto error_return;
    }

    switch (value)
    {
        case 0:
#if FT5x06_DEBUG_VERBOSE
            printk(KERN_DEBUG "%s() - Crosstalk Test Type is now EVEN.\n", __FUNCTION__);
#endif /* FT5x06_DEBUG_VERBOSE */
            ts->crosstalk_test_type = FT5x06_CROSSTALK_TEST_TYPE_EVEN;
            break;

        case 1:
#if FT5x06_DEBUG_VERBOSE
            printk(KERN_DEBUG "%s() - Crosstalk Test Type is now ODD.\n", __FUNCTION__);
#endif /* FT5x06_DEBUG_VERBOSE */
            ts->crosstalk_test_type = FT5x06_CROSSTALK_TEST_TYPE_ODD;
            break;

        default:
            printk(KERN_ERR "%s() - ERROR: Invalid input value specified: %lu\n", __FUNCTION__, value);
            goto error_return;
    }

error_return:
    mutex_unlock(&ts->device_mode_mutex);
    return count;
}


static ssize_t ft5x06_icsupplier_show(struct device *dev, struct device_attribute *attr, char *buf)
{
    struct i2c_client *client = container_of(dev, struct i2c_client, dev);
    struct ft5x06     *ts     = (struct ft5x06 *)i2c_get_clientdata(client);

    ssize_t num_read_chars = 0;
    int     retval = 0;
    u8      regval = 0x00;

    mutex_lock(&ts->device_mode_mutex);
#if FT5x06_DEBUG_VERBOSE
    printk(KERN_DEBUG "%s() - Reading IC Supplier value...\n", __FUNCTION__);
#endif /* FT5x06_DEBUG_VERBOSE */

    retval = i2c_smbus_read_i2c_block_data(ts->client, FT5x06_WMREG_FOCALTECH_ID, sizeof(u8), &regval);
    if (0 > retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not read the IC Supplier value.\n", __FUNCTION__);
        goto error_return;
    }

#if FT5x06_DEBUG_VERBOSE
    printk(KERN_DEBUG "%s() - IC Supplier value: 0x%02X\n", __FUNCTION__, regval);
#endif /* FT5x06_DEBUG_VERBOSE */

    num_read_chars += snprintf(buf, PAGE_SIZE, "0x%02X\n", regval);

error_return:
    mutex_unlock(&ts->device_mode_mutex);
    return num_read_chars;
}


static ssize_t ft5x06_icpartno_show(struct device *dev, struct device_attribute *attr, char *buf)
{
    struct i2c_client *client = container_of(dev, struct i2c_client, dev);
    struct ft5x06     *ts     = (struct ft5x06 *)i2c_get_clientdata(client);

    ssize_t num_read_chars = 0;
    int     retval = 0;
    u8      regval = 0x00;

    mutex_lock(&ts->device_mode_mutex);

    retval = ft5x06_enter_factory_mode(ts);
    if (0 != retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not put the device in Factory Mode.\n", __FUNCTION__);
        goto error_restore_mode;
    }

#if FT5x06_DEBUG_VERBOSE
    printk(KERN_DEBUG "%s() - Reading IC PartNO value...\n", __FUNCTION__);
#endif /* FT5x06_DEBUG_VERBOSE */

    retval = i2c_smbus_read_i2c_block_data(ts->client, FT5x06_FMREG_IC_PARTNO, sizeof(u8), &regval);
    if (0 > retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not read the IC PartNO. value.\n", __FUNCTION__);
        goto error_restore_mode;
    }

#if FT5x06_DEBUG_VERBOSE
    printk(KERN_DEBUG "%s() - IC PartNO value: 0x%02X\n", __FUNCTION__, regval);
#endif /* FT5x06_DEBUG_VERBOSE */

    num_read_chars += snprintf(buf, PAGE_SIZE, "0x%02X\n", regval);

error_restore_mode:
    retval = ft5x06_exit_factory_mode(ts);
    if (0 != retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not return the Touch Panel to Working Mode.\n", __FUNCTION__);
    }

    mutex_unlock(&ts->device_mode_mutex);
    return num_read_chars;
}


static ssize_t ft5x06_storecalibrateflash_show(struct device *dev, struct device_attribute *attr, char *buf)
{
    struct i2c_client *client = container_of(dev, struct i2c_client, dev);
    struct ft5x06     *ts     = (struct ft5x06 *)i2c_get_clientdata(client);

    int retval = 0;
    u8  regval = 0x00;

    mutex_lock(&ts->device_mode_mutex);

    retval = ft5x06_enter_factory_mode(ts);
    if (0 != retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not put the Touch Panel in Factory Mode.\n", __FUNCTION__);
        goto error_restore_mode;
    }

#if FT5x06_DEBUG_VERBOSE
    printk(KERN_DEBUG "%s() - Configuring the Calibration register...\n", __FUNCTION__);
#endif /* FT5x06_DEBUG_VERBOSE */

    regval = FT5x06_CALIBRATE_SAVE_TO_FLASH;
    retval = i2c_smbus_write_i2c_block_data(ts->client, FT5x06_FMREG_CALIBRATE, sizeof(u8), &regval);
    if (0 != retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not write to the Calibration register.\n", __FUNCTION__);
        goto error_restore_mode;
    }

    msleep(1000);

    retval = i2c_smbus_read_i2c_block_data(ts->client, FT5x06_FMREG_DEVICE_MODE, sizeof(u8), &regval);
    if (0 > retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not read the current value of the Device Mode register.\n", __FUNCTION__);
        goto error_restore_mode;
    }

    if (FT5x06_MODE_WORKING != regval)
    {
        printk(KERN_ERR "%s() - ERROR: The Device Mode register contained 0x%02X (expected 0x%02X).\n", __FUNCTION__, regval, FT5x06_MODE_WORKING);
        goto error_restore_mode;
    }

error_restore_mode:
    retval = ft5x06_exit_factory_mode(ts);
    if (0 != retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not return the Touch Panel to Working Mode.\n", __FUNCTION__);
    }

    mutex_unlock(&ts->device_mode_mutex);
    return 0;
}


static ssize_t ft5x06_baseline_show(struct device *dev, struct device_attribute *attr, char *buf)
{
    struct i2c_client *client = container_of(dev, struct i2c_client, dev);
    struct ft5x06     *ts     = (struct ft5x06 *)i2c_get_clientdata(client);

    ssize_t num_read_chars = 0;
    int     retval = 0;
    u8      regval = 0x00;

    mutex_lock(&ts->device_mode_mutex);

    retval = ft5x06_enter_factory_mode(ts);
    if (0 != retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not put the Touch Panel in Factory Mode.\n", __FUNCTION__);
        goto error_restore_mode;
    }

#if FT5x06_DEBUG_VERBOSE
    printk(KERN_DEBUG "%s() - Configuring the Baseline register...\n", __FUNCTION__);
#endif /* FT5x06_DEBUG_VERBOSE */

    regval = FT5x06_BASELINE_ENABLE;
    retval = i2c_smbus_write_i2c_block_data(ts->client, FT5x06_FMREG_BASELINE_ENABLE, sizeof(u8), &regval);
    if (0 != retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not write to the Baseline register.\n", __FUNCTION__);
        goto error_restore_mode;
    }

    msleep(2);

    regval = FT5x06_BASELINE_DISABLE;
    retval = i2c_smbus_read_i2c_block_data(ts->client, FT5x06_FMREG_BASELINE_ENABLE, sizeof(u8), &regval);
    if (0 > retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not read the updated value of the Baseline register.\n", __FUNCTION__);
        goto error_restore_baseline;
    }

    if (FT5x06_BASELINE_ENABLE != (FT5x06_BASELINE_MASK & regval))
    {
        printk(KERN_ERR "%s() - ERROR: The Baseline register contained 0x%02X (expected 0x%02X).\n", __FUNCTION__, (FT5x06_BASELINE_MASK & regval), FT5x06_BASELINE_ENABLE);
        goto error_restore_baseline;
    }

    /* Read raw data */

    retval = ft5x06_read_data(ts->client, buf, PAGE_SIZE, &num_read_chars);
    if (0 != retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not read Raw Data from the Touch Panel.\n", __FUNCTION__);
        goto error_restore_baseline;
    }

error_restore_baseline:
#if FT5x06_DEBUG_VERBOSE
    printk(KERN_DEBUG "%s() - Restoring the Baseline register...\n", __FUNCTION__);
#endif /* FT5x06_DEBUG_VERBOSE */

    regval = FT5x06_BASELINE_DISABLE;
    retval = i2c_smbus_write_i2c_block_data(ts->client, FT5x06_FMREG_BASELINE_ENABLE, sizeof(u8), &regval);
    if (0 != retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not write to the Baseline register.\n", __FUNCTION__);
        goto error_restore_mode;
    }

    msleep(2);

    regval = FT5x06_BASELINE_ENABLE;
    retval = i2c_smbus_read_i2c_block_data(ts->client, FT5x06_FMREG_BASELINE_ENABLE, sizeof(u8), &regval);
    if (0 > retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not read the updated value of the Baseline register.\n", __FUNCTION__);
        goto error_restore_mode;
    }

    if (FT5x06_BASELINE_DISABLE != (FT5x06_BASELINE_MASK & regval))
    {
        printk(KERN_ERR "%s() - ERROR: The Baseline register contained 0x%02X (expected 0x%02X).\n", __FUNCTION__, (FT5x06_BASELINE_MASK & regval), FT5x06_BASELINE_DISABLE);
        goto error_restore_mode;
    }

error_restore_mode:
    retval = ft5x06_exit_factory_mode(ts);
    if (0 != retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not return the Touch Panel to Working Mode.\n", __FUNCTION__);
    }

    mutex_unlock(&ts->device_mode_mutex);
    return num_read_chars;
}


static ssize_t ft5x06_tpfwver_show(struct device *dev, struct device_attribute *attr, char *buf)
{
    struct i2c_client *client = container_of(dev, struct i2c_client, dev);
    struct ft5x06     *ts     = (struct ft5x06 *)i2c_get_clientdata(client);

    ssize_t num_read_chars = 0;
    int     fwver = 0;

    mutex_lock(&ts->device_mode_mutex);
    fwver = i2c_smbus_read_byte_data(ts->client, FT5x06_WMREG_FW_VER);
    if (0 > fwver)
    {
        printk(KERN_ERR "%s() - ERROR: Could not read Firmware version Number.\n", __FUNCTION__);
        goto error_return;
    }

    num_read_chars = snprintf(buf, PAGE_SIZE, "%02X\n", (fwver & 0x000000FF));

error_return:
	mutex_unlock(&ts->device_mode_mutex);
    return num_read_chars;
}

/* Read the vendorID from firmware protected area, need to enter into firmware mode */
static ssize_t ft5x06_vendorid_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct i2c_client *client = container_of(dev, struct i2c_client, dev);
	struct ft5x06     *ts     = (struct ft5x06 *)i2c_get_clientdata(client);

	ssize_t num_read_chars = 0;
	u8     vendorid = 0;
	int     retval   = 0;
	u8  	temp_buffer[4];
	struct i2c_msg msgs[2];

	mutex_lock(&ts->device_mode_mutex);

	retval = ft5x06_enter_fwupdate_mode(ts);
	if (0 != retval) {
		goto error_reset;
	}

	temp_buffer[0] = FT5x06_UPGRADEVER_REG;
	msgs[0].addr = ts->client->addr;
	msgs[0].flags = 0;
	msgs[0].len = 1;
	msgs[0].buf = temp_buffer;

	msgs[1].addr = ts->client->addr;
	msgs[1].flags = I2C_M_RD;
	msgs[1].len = 1;
	msgs[1].buf = &vendorid;

	retval = i2c_transfer(ts->client->adapter, msgs, 2);
	if (0 >= retval) {
		printk(KERN_ERR "%s() - ERROR: Could not read the Upgrad Version from the CTPM.\n",  __FUNCTION__);
		goto error_reset;
	}

    num_read_chars = snprintf(buf, PAGE_SIZE, "0x%02X\n", vendorid);
error_reset:
    printk(KERN_INFO "%s() - Reset the Touch Panel\n", __FUNCTION__);
    ft5x06_reset_panel_via_gpio(ts->platform_data->reset_gpio);
	mutex_unlock(&ts->device_mode_mutex);
    return num_read_chars;
}

static ssize_t ft5x06_calibrate_show(struct device *dev, struct device_attribute *attr, char *buf)
{
    struct i2c_client *client = container_of(dev, struct i2c_client, dev);
    struct ft5x06     *ts     = (struct ft5x06 *)i2c_get_clientdata(client);

    ssize_t num_read_chars = 0;
    int     i       = 0;
    int     retval  = 0;
    u8      regval  = 0x00;
    u8      devmode = 0x00;

    u8  calibration_data[FT5x06_CALIBRATION_NUM_BYTES];

    mutex_lock(&ts->device_mode_mutex);

    retval = ft5x06_enter_factory_mode(ts);
    if (0 != retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not put the device in Factory Mode.\n", __FUNCTION__);
        goto error_restore_mode;
    }

#if FT5x06_DEBUG_VERBOSE
    printk(KERN_DEBUG "%s() - Configuring the Calibration register...\n", __FUNCTION__);
#endif /* FT5x06_DEBUG_VERBOSE */

    regval = FT5x06_CALIBRATE_START;
    retval = i2c_smbus_write_i2c_block_data(ts->client, FT5x06_FMREG_CALIBRATE, sizeof(u8), &regval);
    if (0 != retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not write to the Calibration register.\n", __FUNCTION__);
        goto error_restore_mode;
    }

    msleep(1000);

    /* Configuring the Calibration register should put us back in Working Mode. */

    retval = i2c_smbus_read_i2c_block_data(ts->client, FT5x06_WMREG_DEVICE_MODE, sizeof(u8), &devmode);
    if (0 > retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not read the value of the Device Mode register.\n", __FUNCTION__);
        goto error_restore_mode;
    }

    if (FT5x06_MODE_WORKING != devmode)
    {
        printk(KERN_ERR "%s() - ERROR: The Device Mode register contained 0x%02X (expected 0x%02X).\n", __FUNCTION__, devmode, FT5x06_MODE_WORKING);
        goto error_restore_mode;
    }

    /* Go back to Factory Mode, but don't call ft5x06_enter_factory_mode()
     * since that function will disable the IRQ a second time without
     * re-enabling the IRQ first.
     */
#if FT5x06_DEBUG_VERBOSE
    printk(KERN_DEBUG "%s() - Putting the Touch Panel in Factory Mode...\n", __FUNCTION__);
#endif /* FT5x06_DEBUG_VERBOSE */

    regval = FT5x06_MODE_FACTORY;
    retval = i2c_smbus_write_i2c_block_data(ts->client, FT5x06_WMREG_DEVICE_MODE, sizeof(u8), &regval);
    if (0 != retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not write to the Device Mode register.\n", __FUNCTION__);
        goto error_restore_mode;
    }

    msleep(100);

#if FT5x06_DEBUG_VERBOSE
    printk(KERN_DEBUG "%s() - Verifying that the Touch Panel is in Factory Mode...\n", __FUNCTION__);
#endif /* FT5x06_DEBUG_VERBOSE */

    retval = i2c_smbus_read_i2c_block_data(ts->client, FT5x06_FMREG_DEVICE_MODE, sizeof(u8), &regval);
    if (0 > retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not read from the Device Mode register.\n", __FUNCTION__);
        goto error_restore_mode;
    }

    if ((regval & FT5x06_MODE_MASK) != FT5x06_MODE_FACTORY)
    {
        printk(KERN_ERR "%s() - ERROR: The Touch Panel was not put in Factory Mode. The Device Mode register contains 0x%02X\n", __FUNCTION__, regval);
        retval = FT5x06_ERR_NOT_FACTORY_MODE;
        goto error_restore_mode;
    }

#if FT5x06_DEBUG_VERBOSE
    printk(KERN_DEBUG "%s() - Reading calibration data...\n", __FUNCTION__);
#endif /* FT5x06_DEBUG_VERBOSE */

    for (i = 0; i < FT5x06_CALIBRATION_NUM_BYTES; i++)
    {
        retval = i2c_smbus_read_i2c_block_data(ts->client, (FT5x06_FMREG_RX_0_1_OFFSET + i), sizeof(u8), &calibration_data[i]);
        if (0 > retval)
        {
            printk(KERN_ERR "%s() - ERROR: Could not read offsets for RX %d and %d.\n", __FUNCTION__, (2 * i), ((2 * i) + 1));
            goto error_restore_mode;
        }
    }

    msleep(100);

    /* Each calibration byte holds two offsets that can each be up to 2 chars long. Add two spaces and we need 6 chars to store it. */
    for (i = 0; ((i < FT5x06_CALIBRATION_NUM_BYTES) && ((num_read_chars + 6) < PAGE_SIZE)); i++)
    {
        num_read_chars += sprintf(&(buf[num_read_chars]), "%u %u ", (calibration_data[i] & 0x0F), ((calibration_data[i] >> 4) & 0x0F));
    }

    buf[num_read_chars-1] = '\n';

error_restore_mode:
    retval = ft5x06_exit_factory_mode(ts);
    if (0 != retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not return the Touch Panel to Working Mode.\n", __FUNCTION__);
    }

    mutex_unlock(&ts->device_mode_mutex);
    return num_read_chars;
}


static ssize_t ft5x06_voltage_show(struct device *dev, struct device_attribute *attr, char *buf)
{
    struct i2c_client *client = container_of(dev, struct i2c_client, dev);
    struct ft5x06     *ts     = (struct ft5x06 *)i2c_get_clientdata(client);

    ssize_t num_read_chars = 0;
    int     retval  = 0;
    u8      voltage = 0;

    mutex_lock(&ts->device_mode_mutex);
    retval = ft5x06_enter_factory_mode(ts);
    if (0 != retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not put the device in Factory Mode.\n", __FUNCTION__);
        goto error_restore_mode;
    }

#if FT5x06_DEBUG_VERBOSE
    printk(KERN_DEBUG "%s() - Reading the Device Voltage...\n", __FUNCTION__);
#endif /* FT5x06_DEBUG_VERBOSE */

    retval = i2c_smbus_read_i2c_block_data(ts->client, FT5x06_FMREG_DRIVER_VOLTAGE, sizeof(u8), &voltage);
    if (0 > retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not read the value of the Device Voltage register.\n", __FUNCTION__);
        goto error_restore_mode;
    }

    num_read_chars += snprintf(buf, PAGE_SIZE, "%u\n", (FT5x06_VOLTAGE_MASK & voltage));

error_restore_mode:
    retval = ft5x06_exit_factory_mode(ts);
    if (0 != retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not return the Touch Panel to Working Mode.\n", __FUNCTION__);
    }

    mutex_unlock(&ts->device_mode_mutex);
    return num_read_chars;
}

static ssize_t ft5x06_voltage_store(struct device *dev, struct device_attribute *attr, const char *buf, size_t size)
{
    struct i2c_client *client = container_of(dev, struct i2c_client, dev);
    struct ft5x06     *ts     = (struct ft5x06 *)i2c_get_clientdata(client);

    unsigned long voltage = 0;

    int retval = 0;
    u8  regval = 0x00;

    mutex_lock(&ts->device_mode_mutex);

    retval = strict_strtoul(buf, 16, &voltage);
    if (0 != retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not convert the given input to a number. The given input was: \"%s\"\n", __FUNCTION__, buf);
        goto error_return;
    }

    if (0x07 < voltage)
    {
        printk(KERN_ERR "%s() - ERROR: Invalid value specified: %lu\n", __FUNCTION__, voltage);
        goto error_return;
    }

    retval = ft5x06_enter_factory_mode(ts);
    if (0 != retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not put the device in Factory Mode.\n", __FUNCTION__);
        goto error_restore_mode;
    }

    regval = (u8)(voltage & 0x000000FF);

#if FT5x06_DEBUG_VERBOSE
    printk(KERN_DEBUG "%s() - Writing 0x%02X to the Device Voltage register...\n", __FUNCTION__, regval);
#endif /* FT5x06_DEBUG_VERBOSE */

    retval = i2c_smbus_write_i2c_block_data(ts->client, FT5x06_FMREG_DRIVER_VOLTAGE, sizeof(u8), &regval);
    if (0 != retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not write 0x%02X to the Device Voltage register.\n", __FUNCTION__, regval);
        goto error_restore_mode;
    }

error_restore_mode:
    retval = ft5x06_exit_factory_mode(ts);
    if (0 != retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not return the Touch Panel to Working Mode.\n", __FUNCTION__);
    }

error_return:
    mutex_unlock(&ts->device_mode_mutex);
    return size;
}

static ssize_t ft5x06_interrupttest_show(struct device *dev, struct device_attribute *attr, char *buf)
{
	/* enter the directory below and cat the corresponding file to execute test
       	 * ie.  /sys/class/i2c-dev/i2c-2/device/2-0004
	*/

    struct i2c_client *client = container_of(dev, struct i2c_client, dev);
    struct ft5x06     *ts     = (struct ft5x06 *)i2c_get_clientdata(client);

    const int gpio_num = irq_to_gpio(ts->client->irq);

    int test_result = 0;
    int retval   = 0;
    u8  regval   = 0x00;
    u8  regval1050[10]   = {0x00,0x03,0x0a,0x01,0x41,0x00,0x00,0x00,0x00,0x00};

    printk(KERN_ERR "irq=%d : gpnum=%d\n",ts->client->irq, gpio_num);

    mutex_lock(&ts->device_mode_mutex);

if (gProductId == TSW1050)
{
    retval = i2c_smbus_write_i2c_block_data(ts->client, 0, 10, (u8 *)&regval1050);
    if (0 != retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not write to the Interrupt Toggle register.\n", __FUNCTION__);
     //   goto error_restore_mode;
    }
    else { printk(KERN_ERR "Wrote to touch device\n");}
}
else
{
    retval = ft5x06_enter_factory_mode(ts);
    if (0 != retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not put the device in Factory Mode.\n", __FUNCTION__);
        goto error_restore_mode;
    }
}

#if FT5x06_DEBUG_VERBOSE
    printk(KERN_ERR "%s() - Verifying that the Interrupt GPIO is HIGH...\n", __FUNCTION__);
#endif /* FT5x06_DEBUG_VERBOSE */

    if (!ft5x06_poll_gpio(gpio_num, 1))
    {
        printk(KERN_ERR "%s() - ERROR: The Interrupt GPIO is LOW when it should be HIGH.\n", __FUNCTION__);
    //    goto error_restore_mode;
    }
#if FT5x06_DEBUG_VERBOSE
    else
    {
        printk(KERN_ERR "%s() - The Interrupt GPIO is HIGH.\n", __FUNCTION__);
    }

    printk(KERN_ERR "%s() - Toggling the Interrupt GPIO line...\n", __FUNCTION__);
#endif /* FT5x06_DEBUG_VERBOSE */

    /* Note that the interrupt line can be toggled by writing any value to the INTERRRUPT_TOGGLE register. */
    regval = 0x00;
    retval = i2c_smbus_write_i2c_block_data(ts->client, FT5x06_FMREG_INTERRUPT_TOGGLE, sizeof(u8), &regval);
    if (0 != retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not write to the Interrupt Toggle register.\n", __FUNCTION__);
     //   goto error_restore_mode;
    }

#if FT5x06_DEBUG_VERBOSE
    printk(KERN_ERR "%s() - Verifying that the Interrupt GPIO is LOW...\n", __FUNCTION__);
#endif /* FT5x06_DEBUG_VERBOSE */

    if (!ft5x06_poll_gpio(gpio_num, 0))
    {
        printk(KERN_ERR "%s() - ERROR: The Interrupt GPIO is HIGH when it should be LOW.\n", __FUNCTION__);
      //  goto error_restore_mode;
    }
#if FT5x06_DEBUG_VERBOSE
    else
    {
        printk(KERN_ERR "%s() - The Interrupt GPIO is LOW.\n", __FUNCTION__);
    }

    printk(KERN_ERR "%s() - Toggling the Interrupt GPIO line...\n", __FUNCTION__);
#endif /* FT5x06_DEBUG_VERBOSE */

    regval = 0x00;
    retval = i2c_smbus_write_i2c_block_data(ts->client, FT5x06_FMREG_INTERRUPT_TOGGLE, sizeof(u8), &regval);
    if (0 != retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not write to the Interrupt Toggle register.\n", __FUNCTION__);
       // goto error_restore_mode;
    }

#if FT5x06_DEBUG_VERBOSE
    printk(KERN_ERR "%s() - Verifying that the Interrupt GPIO is HIGH...\n", __FUNCTION__);
#endif /* FT5x06_DEBUG_VERBOSE */

    if (!ft5x06_poll_gpio(gpio_num, 1))
    {
        printk(KERN_ERR "%s() - ERROR: The Interrupt GPIO is LOW when it should be HIGH.\n", __FUNCTION__);
        //goto error_restore_mode;
    }
#if FT5x06_DEBUG_VERBOSE
    else
    {
        printk(KERN_ERR "%s() - The Interrupt GPIO is HIGH.\n", __FUNCTION__);
    }
#endif /* FT5x06_DEBUG_VERBOSE */

    test_result = 1;

error_restore_mode:
  //  retval = ft5x06_exit_factory_mode(ts);
  //  if (0 != retval)
  //  {
  //      printk(KERN_ERR "%s() - ERROR: Could not return the Touch Panel to Working Mode.\n", __FUNCTION__);
  //  }

#if FT5x06_DEBUG_VERBOSE
    printk(KERN_ERR "%s() - Test Result: %s\n", __FUNCTION__, (test_result ? "PASS" : "FAIL"));
#endif /* FT5x06_DEBUG_VERBOSE */
    mutex_unlock(&ts->device_mode_mutex);
    return snprintf(buf, PAGE_SIZE, "%d\n", test_result);
}


static ssize_t ft5x06_tpreset_show(struct device *dev, struct device_attribute *attr, char *buf)
{
    struct i2c_client *client = container_of(dev, struct i2c_client, dev);
    struct ft5x06     *ts     = (struct ft5x06 *)i2c_get_clientdata(client);

    mutex_lock(&ts->device_mode_mutex);
#if FT5x06_DEBUG_VERBOSE
    printk(KERN_DEBUG "%s() - Resetting Touch Panel\n", __FUNCTION__);
#endif /* FT5x06_DEBUG_VERBOSE */

    ft5x06_reset_panel_via_gpio(ts->platform_data->reset_gpio);

	mutex_unlock(&ts->device_mode_mutex);
    return 0;
}


static ssize_t ft5x06_fwupdate_store(struct device *dev, struct device_attribute *attr, const char *buf, size_t size)
{
    struct i2c_client *client = container_of(dev, struct i2c_client, dev);
    struct ft5x06     *ts     = (struct ft5x06 *)i2c_get_clientdata(client);
    const struct firmware * fw_blob = NULL;
    char fw_name[256];

    mutex_lock(&ts->device_mode_mutex);

    // buf contains the newline-terminated firmware file name - remove the newline.
    strncpy(fw_name, buf, size - 1);
    fw_name[size - 1] = '\0';

    printk(KERN_DEBUG "%s() - Processing input file: '%s'\n", __FUNCTION__, fw_name);

    if (request_firmware(&fw_blob, fw_name, dev))
    {
        printk(KERN_ERR "%s() - ERROR: request_firmware failed\n", __FUNCTION__);
        goto error_return;
    }

    if (ft5x06_perform_fw_upgrade(dev, fw_blob->data, fw_blob->size))
    {
        printk(KERN_ERR "%s() - ERROR: Could not update the firmware using '%s'\n", __FUNCTION__, fw_name);
        goto error_return;
    }

error_return:
    if (fw_blob)
    {
        release_firmware(fw_blob);
    }

	mutex_unlock(&ts->device_mode_mutex);
	return size;
}


static ssize_t ft5x06_fwupdate_show(struct device *dev, struct device_attribute *attr, char *buf)
{
    /* place holder for future use */
    return -EPERM;
}


static ssize_t ft5x06_fmreg_store(struct device *dev, struct device_attribute *attr, const char *buf, size_t size)
{
    struct i2c_client *client = container_of(dev, struct i2c_client, dev);
    struct ft5x06     *ts     = (struct ft5x06 *)i2c_get_clientdata(client);
    unsigned long fmreg = 0;
    int retval = 0;

	mutex_lock(&ts->device_mode_mutex);

    retval = strict_strtoul(buf, 16, &fmreg);
    if (0 != retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not convert the given input to a number. The given input was: \"%s\"\n", __FUNCTION__, buf);
        goto error_return;
    }

    if ((FT5x06_FMREG_MAX < fmreg) ||
        (FT5x06_FMREG_RESERVED == fmreg) ||
        ((FT5x06_FMREG_RESERVEDBLK_START <= fmreg) && (FT5x06_FMREG_RESERVEDBLK_END >= fmreg)))
    {
        printk(KERN_ERR "%s() - ERROR: Invalid register specified: %lu\n", __FUNCTION__, fmreg);
        goto error_return;
    }

    ts->factory_mode_register = (u8)(fmreg & 0x000000FF);

error_return:
	mutex_unlock(&ts->device_mode_mutex);
    return size;
}


static ssize_t ft5x06_fmreg_show(struct device *dev, struct device_attribute *attr, char *buf)
{
    struct i2c_client *client = container_of(dev, struct i2c_client, dev);
    struct ft5x06     *ts     = (struct ft5x06 *)i2c_get_clientdata(client);
    return snprintf(buf, PAGE_SIZE, "0x%02X\n", ts->factory_mode_register);
}


static ssize_t ft5x06_fmval_store(struct device *dev, struct device_attribute *attr, const char *buf, size_t size)
{
    struct i2c_client *client = container_of(dev, struct i2c_client, dev);
    struct ft5x06     *ts     = (struct ft5x06 *)i2c_get_clientdata(client);

    unsigned long fmval = 0;

    int retval = 0;
    u8  regval = 0x00;

    mutex_lock(&ts->device_mode_mutex);

    retval = strict_strtoul(buf, 16, &fmval);
    if (0 != retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not convert the given input to a number. The given input was: \"%s\"\n", __FUNCTION__, buf);
        goto error_return;
    }

    if (0xFF < fmval)
    {
        printk(KERN_ERR "%s() - ERROR: Invalid value specified: %lu\n", __FUNCTION__, fmval);
        goto error_return;
    }

    retval = ft5x06_enter_factory_mode(ts);
    if (0 != retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not put the device in Factory Mode.\n", __FUNCTION__);
        goto error_restore_mode;
    }

    regval = (u8)(fmval & 0x000000FF);

#if FT5x06_DEBUG_VERBOSE
    printk(KERN_DEBUG "%s() - Writing 0x%02X to the register at offset 0x%02X...\n", __FUNCTION__, regval, ts->factory_mode_register);
#endif /* FT5x06_DEBUG_VERBOSE */

    retval = i2c_smbus_write_i2c_block_data(ts->client, ts->factory_mode_register, sizeof(u8), &regval);
    if (0 != retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not write 0x%02X to the register at offset 0x%02X.\n", __FUNCTION__, regval, ts->factory_mode_register);
        goto error_restore_mode;
    }

error_restore_mode:
    retval = ft5x06_exit_factory_mode(ts);
    if (0 != retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not return the Touch Panel to Working Mode.\n", __FUNCTION__);
    }

error_return:
    mutex_unlock(&ts->device_mode_mutex);
    return size;
}


static ssize_t ft5x06_fmval_show(struct device *dev, struct device_attribute *attr, char *buf)
{
    struct i2c_client *client = container_of(dev, struct i2c_client, dev);
    struct ft5x06     *ts     = (struct ft5x06 *)i2c_get_clientdata(client);

    ssize_t num_read_chars = 0;
    int     retval = 0;
    u8      regval = 0;

    mutex_lock(&ts->device_mode_mutex);

    retval = ft5x06_enter_factory_mode(ts);
    if (0 != retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not put the device in Factory Mode.\n", __FUNCTION__);
        goto error_restore_mode;
    }

#if FT5x06_DEBUG_VERBOSE
    printk(KERN_DEBUG "%s() - Reading the register at offset 0x%02X...\n", __FUNCTION__, ts->factory_mode_register);
#endif /* FT5x06_DEBUG_VERBOSE */

    retval = i2c_smbus_read_i2c_block_data(ts->client, ts->factory_mode_register, sizeof(u8), &regval);
    if (0 > retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not read the value of the register at offset 0x%02X.\n", __FUNCTION__, ts->factory_mode_register);
        goto error_restore_mode;
    }

#if FT5x06_DEBUG_VERBOSE
    printk(KERN_DEBUG "%s() - The register at offset 0x%02X contains value 0x%02X.\n", __FUNCTION__, ts->factory_mode_register, regval);
#endif /* FT5x06_DEBUG_VERBOSE */

    num_read_chars += snprintf(buf, PAGE_SIZE, "0x%02X\n", regval);

error_restore_mode:
    retval = ft5x06_exit_factory_mode(ts);
    if (0 != retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not return the Touch Panel to Working Mode.\n", __FUNCTION__);
    }

    mutex_unlock(&ts->device_mode_mutex);
    return num_read_chars;
}


static ssize_t ft5x06_wmreg_store(struct device *dev, struct device_attribute *attr, const char *buf, size_t size)
{
    struct i2c_client *client = container_of(dev, struct i2c_client, dev);
    struct ft5x06     *ts     = (struct ft5x06 *)i2c_get_clientdata(client);
    unsigned long wmreg = 0;
    int retval = 0;

    mutex_lock(&ts->device_mode_mutex);
    retval = strict_strtoul(buf, 16, &wmreg);
    if (0 != retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not convert the given input to a number. The given input was: \"%s\"\n", __FUNCTION__, buf);
        goto error_return;
    }

    switch (wmreg)
    {
        case FT5x06_WMREG_DEVICE_MODE:
        case FT5x06_WMREG_GEST_ID:
        case FT5x06_WMREG_TD_STATUS:
        case FT5x06_WMREG_P1_XH:
        case FT5x06_WMREG_P1_XL:
        case FT5x06_WMREG_P1_YH:
        case FT5x06_WMREG_P1_YL:
        case FT5x06_WMREG_P2_XH:
        case FT5x06_WMREG_P2_XL:
        case FT5x06_WMREG_P2_YH:
        case FT5x06_WMREG_P2_YL:
        case FT5x06_WMREG_TH_TOUCH:
        case FT5x06_WMREG_RPT_RATE:
        case FT5x06_WMREG_OFFSET_LEFT_RIGHT:
        case FT5x06_WMREG_DISTANCE_ZOOM:
        case FT5x06_WMREG_LIB_VER_H:
        case FT5x06_WMREG_LIB_VER_L:
        case FT5x06_WMREG_PWR_MODE:
        case FT5x06_WMREG_FW_VER:
        case FT5x06_WMREG_FOCALTECH_ID:
        case FT5x06_WMREG_RESET:
            ts->working_mode_register = (u8)(wmreg & 0x000000FF);
            break;

        default:
            printk(KERN_ERR "%s() - ERROR: Invalid register specified: %lu\n", __FUNCTION__, wmreg);
            goto error_return;
    }

error_return:
    mutex_unlock(&ts->device_mode_mutex);
    return size;
}


static ssize_t ft5x06_wmreg_show(struct device *dev, struct device_attribute *attr, char *buf)
{
    struct i2c_client *client = container_of(dev, struct i2c_client, dev);
    struct ft5x06     *ts     = (struct ft5x06 *)i2c_get_clientdata(client);

    return snprintf(buf, PAGE_SIZE, "0x%02X\n", ts->working_mode_register);
}


static ssize_t ft5x06_wmval_store(struct device *dev, struct device_attribute *attr, const char *buf, size_t size)
{
    struct i2c_client *client = container_of(dev, struct i2c_client, dev);
    struct ft5x06     *ts     = (struct ft5x06 *)i2c_get_clientdata(client);

    unsigned long wmval = 0;

    int retval = 0;
    u8  regval = 0x00;

    mutex_lock(&ts->device_mode_mutex);
    retval = strict_strtoul(buf, 16, &wmval);
    if (0 != retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not convert the given input to a number. The given input was: \"%s\"\n", __FUNCTION__, buf);
        goto error_return;
    }

    if (0xFF < wmval)
    {
        printk(KERN_ERR "%s() - ERROR: Invalid value specified: %lu\n", __FUNCTION__, wmval);
        goto error_return;
    }

    regval = (u8)(wmval & 0x000000FF);

#if FT5x06_DEBUG_VERBOSE
    printk(KERN_DEBUG "%s() - Writing 0x%02X to the register at offset 0x%02X...\n", __FUNCTION__, regval, ts->working_mode_register);
#endif /* FT5x06_DEBUG_VERBOSE */

    retval = i2c_smbus_write_i2c_block_data(ts->client, ts->working_mode_register, sizeof(u8), &regval);
    if (0 != retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not write 0x%02X to the register at offset 0x%02X.\n", __FUNCTION__, regval, ts->working_mode_register);
        goto error_return;
    }

error_return:
    mutex_unlock(&ts->device_mode_mutex);
    return size;
}


static ssize_t ft5x06_wmval_show(struct device *dev, struct device_attribute *attr, char *buf)
{
    struct i2c_client *client = container_of(dev, struct i2c_client, dev);
    struct ft5x06     *ts     = (struct ft5x06 *)i2c_get_clientdata(client);

    ssize_t num_read_chars = 0;
    int     retval = 0;
    u8      regval = 0;

    mutex_lock(&ts->device_mode_mutex);

#if FT5x06_DEBUG_VERBOSE
    printk(KERN_DEBUG "%s() - Reading the register at offset 0x%02X...\n", __FUNCTION__, ts->working_mode_register);
#endif /* FT5x06_DEBUG_VERBOSE */

    retval = i2c_smbus_read_i2c_block_data(ts->client, ts->working_mode_register, sizeof(u8), &regval);
    if (0 > retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not read the value of the register at offset 0x%02X.\n", __FUNCTION__, ts->working_mode_register);
        goto error_return;
    }

#if FT5x06_DEBUG_VERBOSE
    printk(KERN_DEBUG "%s() - The register at offset 0x%02X contains value 0x%02X.\n", __FUNCTION__, ts->working_mode_register, regval);
#endif /* FT5x06_DEBUG_VERBOSE */

    num_read_chars += snprintf(buf, PAGE_SIZE, "0x%02X\n", regval);

error_return:
    mutex_unlock(&ts->device_mode_mutex);
    return num_read_chars;
}


/* sysfs */
static DEVICE_ATTR(version,             S_IRUGO,           ft5x06_version_show,             NULL);
static DEVICE_ATTR(rawbase,             S_IRUGO,           ft5x06_rawbase_show,             NULL);
static DEVICE_ATTR(crosstalk,           S_IRUGO | S_IWUSR, ft5x06_crosstalk_show,           ft5x06_crosstalk_store);
static DEVICE_ATTR(icsupplier,          S_IRUGO,           ft5x06_icsupplier_show,          NULL);
static DEVICE_ATTR(icpartno,            S_IRUGO,           ft5x06_icpartno_show,            NULL);
static DEVICE_ATTR(storecalibrateflash, S_IRUGO,           ft5x06_storecalibrateflash_show, NULL);
static DEVICE_ATTR(baseline,            S_IRUGO,           ft5x06_baseline_show,            NULL);
static DEVICE_ATTR(tpfwver,             S_IRUGO,           ft5x06_tpfwver_show,             NULL);
static DEVICE_ATTR(vendorid,            S_IRUGO,           ft5x06_vendorid_show,            NULL);
static DEVICE_ATTR(voltage,             S_IRUGO | S_IWUSR, ft5x06_voltage_show,             ft5x06_voltage_store);
static DEVICE_ATTR(calibrate,           S_IRUGO,           ft5x06_calibrate_show,           NULL);
static DEVICE_ATTR(interrupttest,       S_IRUGO,           ft5x06_interrupttest_show,       NULL);
static DEVICE_ATTR(tpreset,             S_IRUGO,           ft5x06_tpreset_show,             NULL);
static DEVICE_ATTR(fwupdate,            S_IRUGO | S_IWUSR, ft5x06_fwupdate_show,            ft5x06_fwupdate_store);
static DEVICE_ATTR(fmreg,               S_IRUGO | S_IWUSR, ft5x06_fmreg_show,               ft5x06_fmreg_store);
static DEVICE_ATTR(fmval,               S_IRUGO | S_IWUSR, ft5x06_fmval_show,               ft5x06_fmval_store);
static DEVICE_ATTR(wmreg,               S_IRUGO | S_IWUSR, ft5x06_wmreg_show,               ft5x06_wmreg_store);
static DEVICE_ATTR(wmval,               S_IRUGO | S_IWUSR, ft5x06_wmval_show,               ft5x06_wmval_store);

static struct attribute *ft5x06_attributes[] = {
    &dev_attr_version.attr,
    &dev_attr_rawbase.attr,
    &dev_attr_crosstalk.attr,
    &dev_attr_icsupplier.attr,
    &dev_attr_icpartno.attr,
    &dev_attr_storecalibrateflash.attr,
    &dev_attr_baseline.attr,
    &dev_attr_tpfwver.attr,
    &dev_attr_vendorid.attr,
    &dev_attr_voltage.attr,
    &dev_attr_calibrate.attr,
    &dev_attr_interrupttest.attr,
    &dev_attr_tpreset.attr,
    &dev_attr_fwupdate.attr,
    &dev_attr_fmreg.attr,
    &dev_attr_fmval.attr,
    &dev_attr_wmreg.attr,
    &dev_attr_wmval.attr,
    NULL
};

static struct attribute_group ft5x06_attribute_group = {
    .attrs = ft5x06_attributes
};


/* ************************************************************************
 * Probe and Initialization functions
 * ***********************************************************************/

/* ft5x06_initialize: Driver Initialization. This function takes
 * care of the following tasks:
 * 1. Create and register an input device with input layer
 * 2. Take FT5x06 device out of bootloader mode; go operational
 * 3. Start any timers/Work queues.
 * Note that validation has already been performed on the inputs.
 */
static int ft5x06_initialize(struct i2c_client *client, struct ft5x06 *ts)
{
	struct input_dev *input_device;
	struct input_dev *keyinput_device;
	int retval = 0;
	u8 id;
	int virt_key;
	// Write buffer for the 1050
	u8 versionQuery1050[10] = {0x03,0x03,0x0a,0x01,'D',0,0,0,0,0};
	u8 nameQuery1050[10]    = {0x03,0x03,0x0a,0x01,'E',0,0,0,0,0};
	u8 versionStr[20] = "";
	u8 wbuffer1050[10]={0x03,0x03,0x0a,0x01,0x41,0,0,0,0,0}; // 0x41 = 'A'

	/* Create the input device and register it. */
	input_device = input_allocate_device();
	if (!input_device)
	{
		retval = -ENOMEM;
		printk(KERN_ERR "%s() - ERROR: Could not allocate input device.\n", __FUNCTION__);
		goto error_free_device;
	}


	ts->input = input_device;

	input_device->name = FT_I2C_NAME;
	input_device->phys = ts->phys;
	input_device->dev.parent = &client->dev;

	/* init the touch structures */
	ts->num_prv_st_tch = FT_NTCH;

	for (id = 0; id < FT_NUM_TRK_ID; id++)
	{
		ts->act_trk[id] = FT_NTCH;
		ts->prv_mt_pos[id][FT_XPOS] = 0;
		ts->prv_mt_pos[id][FT_YPOS] = 0;
	}

	for (id = 0; id < FT_NUM_MT_TCH_ID; id++)
	{
		ts->prv_mt_tch[id] = FT_IGNR_TCH;
	}

	for (id = 0; id < FT_NUM_ST_TCH_ID; id++)
	{
		ts->prv_st_tch[id] = FT_IGNR_TCH;
	}

	set_bit(EV_SYN,    input_device->evbit);
	set_bit(EV_KEY,    input_device->evbit);
	set_bit(EV_ABS,    input_device->evbit);

	set_bit(BTN_TOUCH, input_device->keybit);
	set_bit(BTN_2,     input_device->keybit);

	if (ts->platform_data->use_gestures)
	{
		set_bit(BTN_3, input_device->keybit);
	}

	input_set_abs_params(input_device, ABS_X,          0, ts->platform_data->maxx, 0, 0);
	input_set_abs_params(input_device, ABS_Y,          0, ts->platform_data->maxy, 0, 0);
	input_set_abs_params(input_device, ABS_TOOL_WIDTH, 0, FT_LARGE_TOOL_WIDTH,     0, 0);
	input_set_abs_params(input_device, ABS_PRESSURE,   0, FT_MAXZ,                 0, 0);
	input_set_abs_params(input_device, ABS_HAT0X,      0, ts->platform_data->maxx, 0, 0);
	input_set_abs_params(input_device, ABS_HAT0Y,      0, ts->platform_data->maxy, 0, 0);

	if (ts->platform_data->use_gestures)
	{
		input_set_abs_params(input_device, ABS_HAT1X, 0, 255, 0, 0);  // Gesture code.
		input_set_abs_params(input_device, ABS_HAT2X, 0, 255, 0, 0);  // Gesture touch count.
		input_set_abs_params(input_device, ABS_HAT2Y, 0, 255, 0, 0);  // Gesture occur count.
	}

	if (ts->platform_data->use_mt)
	{
		input_set_abs_params(input_device, ABS_MT_POSITION_X,  0, ts->platform_data->maxx, 0, 0);
		input_set_abs_params(input_device, ABS_MT_POSITION_Y,  0, ts->platform_data->maxy, 0, 0);
		input_set_abs_params(input_device, ABS_MT_TOUCH_MAJOR, 0, FT_MAXZ,                 0, 0);
		input_set_abs_params(input_device, ABS_MT_WIDTH_MAJOR, 0, FT_LARGE_TOOL_WIDTH,     0, 0);

		if (ts->platform_data->use_trk_id)
		{
			if (gProductId == TSW1050)
				input_set_abs_params(input_device, ABS_MT_TRACKING_ID, 0, 2, 0, 0);
			else
				input_set_abs_params(input_device, ABS_MT_TRACKING_ID, 0, FT_NUM_TRK_ID, 0, 0);
		}
	}

	input_set_capability(input_device, EV_KEY, KEY_PROG1);

	retval = input_register_device(input_device);
	if (0 != retval)
	{
		printk(KERN_ERR "%s() - ERROR: Could not register input device.\n", __FUNCTION__);
		goto error_free_device;
	}

	/* These are the virtual buttons (need to let android now which ones you will be using) */
	if(ts->platform_data->has_virt_btn) {
		/* Create the input device and register it. */
		keyinput_device = input_allocate_device();
		if (!keyinput_device)
		{
			retval = -ENOMEM;
			printk(KERN_ERR "%s() - ERROR: Could not allocate input device.\n", __FUNCTION__);
			goto error_free_device;
		}
		ts->keyinput = keyinput_device;

		keyinput_device->name = "touchVKey";

		set_bit(EV_KEY,    keyinput_device->evbit);
		/* set_bit(KEY_BACK,     keyinput_device->keybit); */
		/* set_bit(KEY_HOME,     keyinput_device->keybit); */
		/* set_bit(KEY_MENU,     keyinput_device->keybit); */
		/* set_bit(KEY_VOLUMEUP,     keyinput_device->keybit); */
		/* set_bit(KEY_VOLUMEDOWN,     keyinput_device->keybit); */
		for(virt_key=0; virt_key < FT5x06_NUM_VIRT_BUTTON; ++virt_key) {
			set_bit(ts->platform_data->virt_btn[virt_key].keycode, keyinput_device->keybit);
		}
		set_bit(KEY_F14,     keyinput_device->keybit); //UIHAL overload for virtual key event

		retval = input_register_device(keyinput_device);
		if (0 != retval)
		{
			printk(KERN_ERR "%s() - ERROR: Could not register input device.\n", __FUNCTION__);
			input_free_device(keyinput_device);
			goto error_free_device;
		}
	}

	/* Prepare our worker structure prior to setting up the timer/ISR */
	INIT_WORK(&ts->work, ft5x06_xy_worker);

	if (gProductId == TSW1050)
	{
		input_sync(input_device);
	}

	atomic_set(&ts->irq_enabled, 1);

	/* Interrupt setup */
	if (ts->client->irq)
	{
		/* request_irq() will call enable_irq() */
		retval = request_irq(ts->client->irq, ft5x06_irq, IRQF_TRIGGER_FALLING, input_device->name, ts);
		if (retval)
		{
			printk(KERN_ERR "%s() - ERROR: Could not request IRQ: %d\n", __FUNCTION__, retval);
			goto error_free_irq;
		}
	}

	retval = sysfs_create_group(&client->dev.kobj, &ft5x06_attribute_group);
	if (retval)
	{
		printk(KERN_ERR "%s() - ERROR: sysfs_create_group() failed: %d\n", __FUNCTION__, retval);
	}
	else
	{
		printk(KERN_INFO "%s() - sysfs_create_group() succeeded.\n", __FUNCTION__);
	}

	ts->crosstalk_test_type   = FT5x06_CROSSTALK_TEST_TYPE_EVEN;
	ts->factory_mode_register = FT5x06_FMREG_DEVICE_MODE;
	ts->working_mode_register = FT5x06_WMREG_DEVICE_MODE;

	retval = device_create_file(&ts->client->dev, &dev_attr_irq_enable);
	if (retval < 0)
	{
		printk(KERN_ERR "%s() - ERROR: File device creation failed: %d\n", __FUNCTION__, retval);
		retval = -ENODEV;
		goto error_free_irq;
	}

	uiTouchIntfInit(ts);
	if (ts->platform_data->has_virt_btn)
	{
		// do not call this if tst600 since it has its own button uiHal interface
		uiVbuttonIntfInit();
	}

	if(gProductId == TSW1050)
	{
		//TODO: Decide if we want to send an initial firmware test command.
		//CE does not send an initial packet on the i2c.
		//intialize the tsw1050 touch with first pkt or the device will not function properly.
		//retval = i2c_smbus_write_i2c_block_data(ts->client, 0, sizeof(versionQuery1050), (u8 *)&versionQuery1050);
//		retval = i2c_smbus_write_i2c_block_data(ts->client, 0, sizeof(wbuffer1050), (u8 *)&wbuffer1050);
//		if (0 != retval)
//		{
//			//printk(KERN_ERR "%s() - ERROR: Could not write to the Interrupt Toggle register.\n", __FUNCTION__);
//			printk(KERN_ERR "%s() - ERROR: Could not request firmware version.\n", __FUNCTION__);
//			goto error_free_irq;
//		}
		// Read the result
		//EP1010MLA9_readStr(ts, versionStr);
	}
	goto success;

error_free_irq:
	free_irq(ts->client->irq, ts);

error_free_device:
	if (input_device)
	{
		input_free_device(input_device);
		destroy_workqueue(ft5x06_ts_wq);
		ts->input = NULL;
	}

success:
	return retval;
}

/* I2C driver probe function */
static int __devinit ft5x06_ts_probe(struct i2c_client *client, const struct i2c_device_id *id)
{
    struct ft5x06 *ts;
    int retval = 0;
    u8  lbuffer = 0;
    u8 	wbuffer[10];  // this buffer used to write initial pkt to tsw1050 touch

    ts = kzalloc(sizeof(struct ft5x06), GFP_KERNEL);
    if (NULL == ts)
    {
        printk(KERN_ERR "%s() - ERROR: Could not allocate %d bytes of kernel memory for ft5x06 struct.\n", __FUNCTION__, sizeof(struct ft5x06));
        retval = -ENOMEM;
        goto error_devinit0;
    }
#if 0 //TBD: RJK, not working properly
    ts->vtp = regulator_get(NULL, "vtp");
    if (IS_ERR(ts->vtp))
    {
        printk(KERN_ERR "%s() - ERROR: Could not get VTP Regulator.\n", __FUNCTION__);
        retval = IS_ERR(ts->vtp);
        goto error_free_ts;
    }
    retval = regulator_enable(ts->vtp);
    if (retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not enable VTP Regulator.\n", __FUNCTION__);
        goto error_regulator_put;
    }
#endif

    msleep(100);

    ts->client = client;
    ts->platform_data = client->dev.platform_data;
    i2c_set_clientdata(client, ts);

    //register_ft_i2c_adapter(client->adapter); // RJK, done in mx53_loco.c   

    retval = i2c_smbus_read_i2c_block_data(ts->client, 0, sizeof(u8), &lbuffer);
    if (0 > retval)
    {
        printk(KERN_ERR "%s() - ERROR: FT5x06 not found on I2C bus.\n", __FUNCTION__);
        goto error_regulator_disable;
    }

    printk(KERN_INFO "%s() - FT5x06 found on I2C bus.\n", __FUNCTION__);

    ft5x06_reset_panel_via_gpio(ts->platform_data->reset_gpio);
    

    // Need to initialize the SYSFS mutex before creating the SYSFS entries in ft5x06_initialize().
    mutex_init(&ts->device_mode_mutex);
    keyStateMutex = kmalloc(sizeof(struct mutex), GFP_KERNEL);
    mutex_init(keyStateMutex);

    retval = ft5x06_initialize(client, ts);
    if (0 > retval)
    {
        printk(KERN_ERR "%s() - ERROR: Controller could not be initialized.\n", __FUNCTION__);
        goto error_mutex_destroy;
    }

#ifdef CONFIG_HAS_EARLYSUSPEND
    ts->early_suspend.level   = EARLY_SUSPEND_LEVEL_BLANK_SCREEN + 1;
    ts->early_suspend.suspend = ft5x06_early_suspend;
    ts->early_suspend.resume  = ft5x06_late_resume;

    register_early_suspend(&ts->early_suspend);
#endif /* CONFIG_HAS_EARLYSUSPEND */

    goto error_return;

error_devinit0:
error_mutex_destroy:
    mutex_destroy(&ts->device_mode_mutex);
error_regulator_disable:
   regulator_disable(ts->vtp);

error_regulator_put:
   regulator_put(ts->vtp);

error_free_ts:
    kfree(ts);

error_return:
    return retval;
}


/* Function to manage power-on resume */
static int ft5x06_resume(struct i2c_client *client)
{
    int retval = 0;

    printk(KERN_ERR "%s() - Driver is resuming.\n", __FUNCTION__);
#if 0 //RJK not supported yet.
    struct ft5x06 *ts = NULL;

    ts = (struct ft5x06 *) i2c_get_clientdata(client);
    retval = regulator_enable(ts->vtp);
    if (retval)
    {
        printk(KERN_ERR "%s() - ERROR: Could not enable regulator.\n", __FUNCTION__);
    }
    else
    {
		if (ts->platform_data->platform_resume) {
			ts->platform_data->platform_resume();
		}
        	ft5x06_reset_panel_via_gpio(ts->platform_data->reset_gpio);
		mb();
		enable_irq(ts->client->irq);
    }
#endif
    return retval;
}


/* Function to manage low power suspend */
static int ft5x06_suspend(struct i2c_client *client, pm_message_t message)
{
    int retval = 0;

    struct ft5x06 *ts = NULL;

    printk(KERN_ERR "%s() - Driver is suspending.\n", __FUNCTION__);

#if 0 //RJK not supported yet
    ts = (struct ft5x06 *) i2c_get_clientdata(client);

   /* Disable/enable irq call (in irq/worker function) are matched, disable here for suspend */
    disable_irq(ts->client->irq);

   /* Wait for woker finish, even if worker enables irq, the irq still is disabled because of the above call */
    flush_workqueue(ft5x06_ts_wq);

   /* No need to cancel since irq is disabled, there is no pending worker at this time*/

	if (ts->platform_data->platform_suspend) {
		ts->platform_data->platform_suspend();
	}
	// keep focaltech controller in reset after this point
    gpio_direction_output(ts->platform_data->reset_gpio, 0);
    regulator_disable(ts->vtp);
#endif //RJK 

    return 0;
}


/* registered in driver struct */
static int __devexit ft5x06_ts_remove(struct i2c_client *client)
{
    struct ft5x06 *ts;

    printk(KERN_INFO "%s() - Driver is unregistering.\n", __FUNCTION__);

    /* clientdata registered on probe */
    ts = i2c_get_clientdata(client);
    device_remove_file(&ts->client->dev, &dev_attr_irq_enable);

 //RJK - may need to add this back   // unregister_ft_i2c_adapter(client->adapter);

    /* Start cleaning up by removing any delayed work and the timer */
    if (cancel_delayed_work_sync((struct delayed_work *)&ts->work) < 0)
    {
        printk(KERN_ERR "%s() - ERROR: Could not remove all work from the Work Queue.\n", __FUNCTION__);
    }

    /* free up timer or irq */
    if (ts->client->irq == 0)
    {
    }
    else
    {
        free_irq(client->irq, ts);
    }

#ifdef CONFIG_HAS_EARLYSUSPEND
    unregister_early_suspend(&ts->early_suspend);
#endif /* CONFIG_HAS_EARLYSUSPEND */

	regulator_disable(ts->vtp);
	regulator_put(ts->vtp);

    /* housekeeping */
    /* Wait until any outstanding SYSFS transaction has finished,
     * and prevent any new ones from starting.
     */
    mutex_lock(&ts->device_mode_mutex);
    /* Remove the SYSFS entries */
    sysfs_remove_group(&client->dev.kobj, &ft5x06_attribute_group);
    mutex_unlock(&ts->device_mode_mutex);
    mutex_destroy(&ts->device_mode_mutex);

    if (NULL != ts)
    {
        kfree(ts);
    }

    printk(KERN_INFO "%s() - Driver unregistration is complete.\n", __FUNCTION__);

    return 0;
}


#ifdef CONFIG_HAS_EARLYSUSPEND
static void ft5x06_early_suspend(struct early_suspend *handler)
{
    struct ft5x06 *ts;
    ts = container_of(handler, struct ft5x06, early_suspend);
    ft5x06_suspend(ts->client, PMSG_SUSPEND);
}

static void ft5x06_late_resume(struct early_suspend *handler)
{
    struct ft5x06 *ts;
    ts = container_of(handler, struct ft5x06, early_suspend);
    ft5x06_resume(ts->client);
}
#else  /* CONFIG_HAS_EARLYSUSPEND */
static void ft5x06_early_suspend(struct early_suspend *handler)
{
    /* Do Nothing */
}

static void ft5x06_late_resume(struct early_suspend *handler)
{
    /* Do Nothing */
}
#endif /* CONFIG_HAS_EARLYSUSPEND */


static int ft5x06_ts_init(void)
{
    int ret = 0;

    printk(KERN_INFO "%s() - FT5x06 I2C Touchscreen Driver (Built %s @ %s)\n", __FUNCTION__, __DATE__, __TIME__);

    ft5x06_ts_wq = create_singlethread_workqueue("ft5x06_ts_wq");
    if (NULL == ft5x06_ts_wq)
    {
        printk(KERN_ERR "%s() - ERROR: Could not create the Work Queue due to insufficient memory.\n", __FUNCTION__);
        ret = -ENOMEM;
    }
    else
    {
if(gProductId == TST600 || gProductId == TST900 )
        ret = i2c_add_driver(&ft5x06_ts_driver);
if(gProductId == TSW1050 || gProductId == TSW750 || gProductId == TSW550 || gProductId == TSM750)
        ret = i2c_add_driver(&ft5x06_ts_driver_1050);
    }

    return ret;
}


static void ft5x06_ts_exit(void)
{
 
    if (ft5x06_ts_wq)
    {
        destroy_workqueue(ft5x06_ts_wq);
    }
if(gProductId == TST600 || gProductId == TST900 )
	i2c_del_driver(&ft5x06_ts_driver);
if(gProductId == TSW1050 || gProductId == TSW750 || gProductId == TSW550 || gProductId == TSM750 )
	i2c_del_driver(&ft5x06_ts_driver_1050);

}

module_init(ft5x06_ts_init);
module_exit(ft5x06_ts_exit);

